/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 EADS - CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.portlet.ActionRequest;
import javax.portlet.ActionResponse;
import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Results;
import org.ow2.weblab.core.helper.impl.SimpleSelector;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.portlet.bean.FacetBean;
import org.ow2.weblab.portlet.bean.FacetValueBean;
import org.ow2.weblab.portlet.bean.SelectedFacetValueBean;
import org.ow2.weblab.portlet.conf.FacetConfigBean;
import org.ow2.weblab.portlet.tool.FacetUtil;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.xml.XmlBeanFactory;
import org.springframework.core.io.ClassPathResource;

public class FacetPortlet extends WebLabPortlet {

	private final static Log logger = LogFactory.getLog(FacetPortlet.class);

	/**
	 * Facet selection action
	 */
	public final static String FACET_SELECTION_ACTION = "facetSelection";

	/**
	 * Facet unselection action
	 */
	public final static String FACET_UNSELECTION_ACTION = "facetUnselection";

	/**
	 * User Facet filters list session property
	 */
	public final static String USER_FACET_FILTERS_MAP = "facetFiltersMap";

	/**
	 * User Facet selected values displayed on JSP
	 */
	public final static String USER_SELECTED_FACET_MAP = "selectedFacetMap";

	
	/**
	 *  Facet display styles on JSP
	 */
	public final static String FACET_DISPLAY_STYLES = "facetDisplayStyles";
	
	/**
	 *  Colors used for Pie display on JSP
	 */
	public final static String FACET_PIE_COLORS = "facetPieColors";
	
	/**
	 * Query URI
	 */
	public final static String LAST_QUERY_URI = "lastQueryURI";

	/**
	 * Attribute name used to send an error message to jsp
	 */
	public static final String ERROR = "message_error";

	/**
	 * Attribute name of the user's current ResultSet in session
	 */
	private static final String USER_RESULT_SET = "resultSet";
	private static final String USER_FACET_COLLECTION = "facetCollection";
	private static final String SHOW_FILTER = "showFilter";
	private static final boolean SHOW_FILTER_VALUE = false;

	protected FacetConfigBean portletConf;

	@Override
	public void init() throws PortletException {
		super.init();

		ClassPathResource resource = new ClassPathResource(FacetConfigBean.DEFAULT_CONF_FILE);
		BeanFactory factory = new XmlBeanFactory(resource);
		this.portletConf = factory.getBean(FacetConfigBean.DEFAULT_BEAN_NAME, FacetConfigBean.class);
	}

	@SuppressWarnings("unchecked")
	@Override
	protected void doView(RenderRequest request, RenderResponse response) throws PortletException, IOException {

		if (request.getParameter(ERROR) != null) {
			/*
			 * dispatch to error jsp
			 */
			request.setAttribute(ERROR, request.getParameter(ERROR));
			PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(getInitParameter("error_page_url"));
			rd.include(request, response);
		}

		ResultSet resultSet = (ResultSet) request.getPortletSession().getAttribute(USER_RESULT_SET,
				PortletSession.APPLICATION_SCOPE);

		if (resultSet != null) {
			/*
			 * Check if query has changed and reset filters map if necessary
			 */
			SimpleSelector selector = RDFSelectorFactory.getSelector();
			Results statements = selector.select(resultSet, WebLabRetrieval.IS_RESULT_OF);
			String queryURI = statements.getTypedValue(resultSet.getUri(), WebLabRetrieval.IS_RESULT_OF, String.class);

			if (request.getPortletSession().getAttribute(LAST_QUERY_URI) != null
					&& !((String) request.getPortletSession().getAttribute(LAST_QUERY_URI)).equals(queryURI)) {
				if (request.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP) != null) {
					request.getPortletSession().setAttribute(USER_FACET_FILTERS_MAP,
							new HashMap<String, LinkedHashMap<String, SelectedFacetValueBean>>());
				}
			}

			/*
			 * Store query URI into session
			 */
			request.getPortletSession().setAttribute(LAST_QUERY_URI, queryURI);

			logger.debug("Get facets for resultSet...");
			
			List<FacetBean> facetCollection = FacetUtil.getFacetsFromResultSet(resultSet, this.portletConf);
			
			
			
			if (facetCollection.isEmpty()) {
				logger.debug("No facet for this resultSet...");
				/*
				 * dispatch to no data jsp
				 */
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(getInitParameter("no_data_page_url"));
				rd.include(request, response);
			} else {
				logger.debug("ResultSet has facets...");
				request.setAttribute(USER_FACET_COLLECTION, facetCollection);
	
				if (request.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP) != null) {
					Map<String, LinkedHashMap<String, SelectedFacetValueBean>> userQueryFilters = (Map<String, LinkedHashMap<String, SelectedFacetValueBean>>) request
							.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP);
	
					// Update count value for selected facets and reorder facets
					Map<String, LinkedHashMap<String, SelectedFacetValueBean>> userQueryFiltersToDisplay = new LinkedHashMap<String, LinkedHashMap<String, SelectedFacetValueBean>>();
					for (String facetFieldToDisplay : this.portletConf.getFacetFieldsList()) {
						if (userQueryFilters.containsKey(facetFieldToDisplay)) {
							userQueryFiltersToDisplay.put(facetFieldToDisplay, userQueryFilters.get(facetFieldToDisplay));
							for (Entry<String, SelectedFacetValueBean> selectedValue : userQueryFiltersToDisplay.get(
									facetFieldToDisplay).entrySet()) {
								for (FacetBean facet : facetCollection) {
									if (facet.getLabel().equals(facetFieldToDisplay)) {
										boolean countValueUpdated = false;
										Iterator<FacetValueBean> facetValuesIterator = facet.getValues().iterator();
										while (facetValuesIterator.hasNext()) {
											FacetValueBean facetValue = facetValuesIterator.next();
											if (selectedValue.getKey().equals(facetValue.getFilterQuery())) {
												selectedValue.getValue().setCount(facetValue.getCount());
												facetValuesIterator.remove();
												countValueUpdated = true;
											}
										}
	
										if (!countValueUpdated)
											selectedValue.getValue().setCount(0);
									}
								}
							}
						}
					}
					request.setAttribute(USER_SELECTED_FACET_MAP, userQueryFiltersToDisplay);
				}
				else {
					request.setAttribute(USER_SELECTED_FACET_MAP, new HashMap<String, LinkedHashSet<SelectedFacetValueBean>>());
				}
				
				request.setAttribute(FACET_DISPLAY_STYLES, this.portletConf.getFacetFieldsVsStylesList());
				request.setAttribute(FACET_PIE_COLORS, this.portletConf.getFacetPieColorsList());

				request.setAttribute(SHOW_FILTER, SHOW_FILTER_VALUE);
	
				/*
				 * dispatch to jsp
				 */
				PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
						getInitParameter("facet_view_page_url"));
				rd.include(request, response);
			}
		}
		else {
			/*
			 * dispatch to no data jsp
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("no_data_page_url"));
			rd.include(request, response);
		}
	}

	/**
	 * handle the incoming event
	 */
	@Override
	public void processEvent(EventRequest req, EventResponse resp) throws PortletException, IOException {
		/*
		 * checking event
		 */
		if (req.getEvent() != null && req.getEvent().getValue() instanceof ResultSet) {
			if (getReaction(req.getEvent().getQName()).getLocalPart().equals("displayResults")) {
				ResultSet receveidResultSet = ((ResultSet) req.getEvent().getValue());	
				logger.info("Event received with ResultSet: " + receveidResultSet);

				/*
				 * adding result set in user session
				 */
				req.getPortletSession().setAttribute(USER_RESULT_SET, receveidResultSet,
						PortletSession.APPLICATION_SCOPE);
			} else if (getReaction(req.getEvent().getQName()).getLocalPart().equals("resetFilter")) {
				req.getPortletSession().removeAttribute(USER_FACET_FILTERS_MAP);
			}
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public void processAction(ActionRequest request, ActionResponse response) throws PortletException, IOException {
		String action = request.getParameter("action");
		String facetQueryFilter = request.getParameter("facet_query_filter");
		String facetQueryFilterOperator = request.getParameter("facet_query_filter_operator");
		String facetValueLabel = request.getParameter("facet_value_label");
		String facetLabel = request.getParameter("facet_label");
		if (action != null
				&& facetQueryFilter != null
				&& facetQueryFilterOperator != null
				&& facetValueLabel != null
				&& facetLabel != null) {
			
			logger.info("Facet action: " + action + " -> " + facetQueryFilterOperator + " " + facetQueryFilter);

			/*
			 * Get query URI
			 */
			ResultSet rs = (ResultSet) request.getPortletSession().getAttribute(USER_RESULT_SET,
					PortletSession.APPLICATION_SCOPE);
			if (rs != null) {
				SimpleSelector selector = RDFSelectorFactory.getSelector();
				Results statements = selector.select(rs, WebLabRetrieval.IS_RESULT_OF);
				String queryURI = statements.getTypedValue(rs.getUri(), WebLabRetrieval.IS_RESULT_OF, String.class);

				logger.info("send facet action for query: " + queryURI);

				PieceOfKnowledge pok = ResourceFactory.createResource("facetporlet", "" + System.nanoTime(),
						PieceOfKnowledge.class);
				PoKHelper h = new JenaPoKHelper(pok);
				h.setAutoCommitMode(false);
				h.createLitStat(pok.getUri(), WebLabRetrieval.HAS_ORDERED_QUERY, queryURI);
				h.createLitStat(pok.getUri(), this.portletConf.getFacetFilterQueryProperty(), facetQueryFilterOperator
						+ " " + facetQueryFilter);
				h.commit();

				/*
				 * store facet query filter to session
				 */
				Map<String, LinkedHashMap<String, SelectedFacetValueBean>> facetFiltersMap;
				if (request.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP) != null) {
					if (request.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP) instanceof Map<?, ?>)
						facetFiltersMap = (Map<String, LinkedHashMap<String, SelectedFacetValueBean>>) request
								.getPortletSession().getAttribute(USER_FACET_FILTERS_MAP);
					else {
						logger.warn("Facet filter list in session was not a 'List' type, reinitialise it.");
						facetFiltersMap = new HashMap<String, LinkedHashMap<String, SelectedFacetValueBean>>();
						request.getPortletSession().setAttribute(USER_FACET_FILTERS_MAP, facetFiltersMap);
					}
				}
				else {
					facetFiltersMap = new HashMap<String, LinkedHashMap<String, SelectedFacetValueBean>>();
					request.getPortletSession().setAttribute(USER_FACET_FILTERS_MAP, facetFiltersMap);
				}
				SelectedFacetValueBean facetValueToAdd = new SelectedFacetValueBean();
				facetValueToAdd.setFilterQuery(facetQueryFilter);
				facetValueToAdd.setOperator(facetQueryFilterOperator);
				facetValueToAdd.setLabel(facetValueLabel);

				if (facetFiltersMap.get(facetLabel) == null)
					facetFiltersMap.put(facetLabel, new LinkedHashMap<String, SelectedFacetValueBean>());

				/*
				 * sending event
				 */
				if (request.getParameter("action").equals("add_facet_filter")) {
					facetFiltersMap.get(facetLabel).put(facetQueryFilter, facetValueToAdd);
					sendEventForAction(FACET_SELECTION_ACTION, pok, response);
				}
				else if (request.getParameter("action").equals("remove_facet_filter")) {
					facetFiltersMap.get(facetLabel).remove(facetQueryFilter);
					// remove category if empty
					if (facetFiltersMap.get(facetLabel).size() == 0)
						facetFiltersMap.remove(facetLabel);

					sendEventForAction(FACET_UNSELECTION_ACTION, pok, response);
				}
			}
			else {
				logger.error("Cannot filter on empty ResultSet.");
			}
		}
	}	
}
