/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 EADS - CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.portlet.tool;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Results;
import org.ow2.weblab.core.helper.impl.SimpleSelector;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.portlet.bean.FacetBean;
import org.ow2.weblab.portlet.bean.FacetValueBean;
import org.ow2.weblab.portlet.conf.FacetConfigBean;

public class FacetUtil {

	private final static Log logger = LogFactory.getLog(FacetUtil.class);

	public static List<FacetBean> getFacetsFromResultSet(ResultSet resultSet, FacetConfigBean portletConf) {
		Map<String, FacetBean> facetMap = new HashMap<String, FacetBean>();

		SimpleSelector selector = RDFSelectorFactory.getSelector();
		Results statements = selector.select(resultSet);

		Set<String> facetURIList = statements.getSubjects(RDF.TYPE, portletConf.getFacetClass());
		Set<String> facetValueURIList = statements.getSubjects(RDF.TYPE, portletConf.getFacetValueClass());

		// Create facet beans
		for (String facetURI : facetURIList) {
			String facetLabel = statements.getTypedValue(facetURI, RDFS.LABEL, String.class);
			if (portletConf.getFacetFieldsList().contains(facetLabel)) {
				FacetBean facetBean = new FacetBean();
				facetBean.setLabel(facetLabel);
				facetBean.setPosition(portletConf.getFacetFieldsList().indexOf(facetLabel));
				facetBean.setValues(new ArrayList<FacetValueBean>());
				facetMap.put(facetURI, facetBean);

				logger.debug("Add facet bean: " + facetBean);
			}
		}

		// Fill facet bean with facet values
		for (String facetValueURI : facetValueURIList) {
			FacetValueBean facetValueBean = new FacetValueBean();
			facetValueBean.setLabel(statements.getTypedValue(facetValueURI, RDFS.LABEL, String.class));
			facetValueBean.setFilterQuery(statements.getTypedValue(facetValueURI, portletConf.getFacetFilterQueryProperty(), String.class));

			String count = statements.getTypedValue(facetValueURI, portletConf.getFacetCountProperty(), String.class);
			if (count != null && !count.isEmpty())
				facetValueBean.setCount(Integer.parseInt(count));
			else
				facetValueBean.setCount(0);

			String linkedFacetURI = statements.getTypedValue(facetValueURI, portletConf.getIsLinkedToFacet(), String.class);

			if (facetMap.containsKey(linkedFacetURI)) {
				facetMap.get(linkedFacetURI).getValues().add(facetValueBean);
			}

			logger.debug("Add facet value: " + facetValueBean + " to facet " + linkedFacetURI);
		}

		// Order facets
		List<FacetBean> facetList = new ArrayList<FacetBean>(facetMap.values());
		Collections.sort(facetList, new FacetBeanComparator());

		// Order facet values for each facets
		FacetValueBeanComparator valueComp = new FacetValueBeanComparator();
		for (FacetBean facet : facetList) {
			Collections.sort(facet.getValues(), valueComp);
		}

		return facetList;
	}

	/**
	 * Normalise Label by settings all first letter in capital and the rest in
	 * lower case.
	 * 
	 * @param s
	 * @return
	 */
	public static String normaliseLabel(String s) {
		final StringBuilder result = new StringBuilder(s.length());
		String[] words = s.split("\\s");
		for (int i = 0, l = words.length; i < l; ++i) {
			if (words[i] != null && words[i].length() > 0) {
				if (i > 0)
					result.append(" ");
				result.append(Character.toUpperCase(words[i].charAt(0))).append(words[i].substring(1).toLowerCase());
			}
		}
		return result.toString();
	}
}
