package org.ow2.weblab.portlet;
/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 *
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of the
 * GNU Lesser General Public License as published by the Free Software Foundation; either version
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA
 */


import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;

import javax.portlet.EventRequest;
import javax.portlet.EventResponse;
import javax.portlet.PortletException;
import javax.portlet.PortletRequestDispatcher;
import javax.portlet.PortletSession;
import javax.portlet.RenderRequest;
import javax.portlet.RenderResponse;
import javax.portlet.ResourceRequest;
import javax.portlet.ResourceResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.impl.AdvancedSelector;
import org.ow2.weblab.core.helper.impl.RDFSelectorFactory;
import org.ow2.weblab.core.helper.impl.Statements;
import org.ow2.weblab.core.helper.impl.WTriple;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.ResourceContainer;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceArgs;
import org.ow2.weblab.portlet.bean.AnnotationsDescBean;
import org.ow2.weblab.portlet.bean.MetaConfBean;
import org.ow2.weblab.portlet.bean.RepoServiceConfigBean;
import org.ow2.weblab.portlet.tool.ResourceContainerUtil;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.xml.XmlBeanFactory;
import org.springframework.core.io.ClassPathResource;


/**
 * A portlet to display WebLab resources metadata.
 * @author Emilien
 *
 */
public class MetaDataPortlet extends WebLabPortlet {

	
	/**
	 * Namespace property used in PieceOkKnowledge to get the Document to
	 * display URI
	 */
	private static final String POK_PROPERTY_DOC_URI = WebLabRetrieval.IS_LINKED_TO;
	
	
	// Private fields used for session
	/**
	 * User's media unit identifier in session
	 */
	private final static String USER_MEDIA_UNIT = "user_media_unit";

	private static final String USER_MEDIA_UNIT_WTMAP = "user_mediaunit_triplemap";
	
	/**
	 * Attribute name of the user's current display configuration bean in
	 * session
	 */
	private static final String META_CONF_BEAN_ID = "meta_conf_bean";


	/**
	 * Attribute name used to send hit description to jsp
	 */
	public static final String META_MAP_DESC = "resource_desc";
	
	/**
	 * Default meta configuration bean
	 */
	public static MetaConfBean DEFAULT_META_CONF_BEAN;

	/**
	 * Attribute name used to send display configuration bean to jsp
	 */
	public static final String USER_META_CONF = "meta_conf_bean";

	private static final String USER_RES_ANNOTS_LOADED = "user_wtriple_map_isloaded";
	
	private static final String USER_MEDIA_UNIT_URI = "resource_uri";
	
	/**
	 * Attribute name used to send repository service configuration bean to jsp
	 */
	public static final String USER_REPO_CONF = "user_repo_service_conf";

	/**
	 * Attribute name used to send a error message to jsp
	 */
	public static final String ERROR = "message_error";
	
	/**
	 * Variable identifier in session to know if the user's current document is
	 * loaded or not
	 */
	private final static String CURRENT_RESSOURCE_IS_LOADED = "user_resource_is_loaded";


	
	/**
	 * Map of resource container services
	 */
	private Map<URL, ResourceContainer> repoMap;

	/**
	 * Default AnnotationsDescBean
	 */
	private AnnotationsDescBean rdfProperties;
	
	/**
	 * Logger
	 */
	private Log logger;
	
	/**
	 * URL of WSDL of repository service
	 */
	private static URL WSDL_REPO_URL;

	

	@Override
	public void init() throws PortletException {
		super.init();
		/*
		 * creating map of repo service
		 */
		this.repoMap = new HashMap<URL, ResourceContainer>();

		/*
		 * getting repository WSDL URL
		 */
		try {

			WSDL_REPO_URL = new File(getPortletContext().getRealPath(
					"WEB-INF/classes/services/WebLab.wsdl")).toURI()
					.toURL();

		} catch (MalformedURLException e) {
			logger.error(e);
		}

		/*
		 * getting meta-data description list from a XML bean, using Spring
		 */
		ClassPathResource resource = new ClassPathResource("conf_meta.xml");
		BeanFactory factory = new XmlBeanFactory(resource);
          
		this.rdfProperties = factory.getBean("metaConf", AnnotationsDescBean.class);
		
		
		/*
		 * initializing meta conf factory
		 */
		DEFAULT_META_CONF_BEAN = factory.getBean(META_CONF_BEAN_ID, MetaConfBean.class);
		
		/*
		 * initializing logger
		 */
		logger = LogFactory.getLog(this.getClass());
	}

	@Override
	public void destroy() {
		this.repoMap = null;
		this.rdfProperties = null;
		
		super.destroy();
	}

	/**
	 * Render HTML corresponding to the view mode
	 */
	@Override
	public void doView(RenderRequest req, RenderResponse res)
			throws IOException, PortletException {

		long start = System.currentTimeMillis();
		boolean sessionExpired = false;
		/*
		 * changing portlet title
		 */
		res.setTitle(ResourceBundle
				.getBundle("meta_portlet", req.getLocale()).getString(
						"portlet.title"));

		/*
		 * checking for session expiration
		 */
		if ((req.getPortletSession().getLastAccessedTime() + (req
				.getPortletSession().getMaxInactiveInterval() * 1000)) < System
				.currentTimeMillis()) {
			sessionExpired = true;
			req.setAttribute("message_warning", ResourceBundle.getBundle(
					"meta_portlet", req.getLocale()).getString(
					"portlet.warning.session_expired"));
		}

		if (!sessionExpired) {
			if (req.getParameter(ERROR) != null && !req.getParameter(ERROR).equals("")) {
				/*
				 * dispatch to error jsp
				 */
				
				logger.error(ERROR);
				req.setAttribute(ERROR, req.getParameter(ERROR));
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(
								getInitParameter("error_page_url"));
				rd.include(req, res);
				
			}
			
			/*
			 * injecting configuration bean
			 */
			req.setAttribute(USER_MEDIA_UNIT, req.getPortletSession().getAttribute(USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE));
		

			/*
			 * dispatch to jsp
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("view_page_url"));
			rd.include(req, res);
		} else {
			/*
			 * Dispatch warning to JSP page
			 */
			PortletRequestDispatcher rd = getPortletContext()
					.getRequestDispatcher(getInitParameter("warning_page_url"));
			rd.include(req, res);
		}
		logger.info(
				"doView reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	@Override
	public void doEdit(RenderRequest req, RenderResponse res)
			throws PortletException, IOException {

		long start = System.currentTimeMillis();
		MetaConfBean metaConfbean;
		RepoServiceConfigBean repoConfBean;
		
		/*
		 * meta conf bean
		 */
		
		
		if (req.getPortletSession().getAttribute(USER_META_CONF,
				PortletSession.APPLICATION_SCOPE) == null) {
			metaConfbean = DEFAULT_META_CONF_BEAN;
		} else {
			metaConfbean = (MetaConfBean) req.getPortletSession().getAttribute(USER_META_CONF,
					PortletSession.APPLICATION_SCOPE);
		}
		/*
		 * injecting configuration bean
		 */
		req.setAttribute(USER_META_CONF, metaConfbean);
		

		/*
		 * repository conf bean
		 */
		if (req.getPortletSession().getAttribute(USER_REPO_CONF,
				PortletSession.APPLICATION_SCOPE) == null) {
			repoConfBean = new RepoServiceConfigBean();
		} else {	
			repoConfBean = (RepoServiceConfigBean) req.getPortletSession().getAttribute(USER_REPO_CONF,
					PortletSession.APPLICATION_SCOPE);
		}
		/*
		 * injecting configuration bean
		 */
		req.setAttribute(USER_REPO_CONF, repoConfBean);
		

		/*
		 * dispatch to jsp
		 */
		PortletRequestDispatcher rd = getPortletContext().getRequestDispatcher(
				getInitParameter("edit_page_url"));

		rd.include(req, res);
		logger.info(
				"doEdit reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	public void processEvent(EventRequest req, EventResponse resp)
	throws PortletException, IOException {
		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();
		
		/*
		 * document received
		 */
		Document doc_evt = null;
		
		/*
		 * checking action according portlet.xml
		 */
		if (getReaction(req.getEvent().getQName()).getLocalPart().equals("displayMeta")) {
			// receive a event containing a document
		
			/*
			 * Setting IS_NOT_LOADED to true because the document is already
			 * loaded
			 */
			req.getPortletSession().setAttribute(CURRENT_RESSOURCE_IS_LOADED,
					true, PortletSession.APPLICATION_SCOPE);
		
			doc_evt = (Document) req.getEvent().getValue();
		
			logger.info("receive a loaded document");
		} else if (getReaction(req.getEvent().getQName()).getLocalPart().equals("loadAndDisplayMeta")) {
			// receive a event containg a PieceOfKnowledge about a document
		
			/*
			 * Setting IS_NOT_LOADED to false because the document is not loaded
			 * yet
			 */
			req.getPortletSession().setAttribute(CURRENT_RESSOURCE_IS_LOADED,
					false, PortletSession.APPLICATION_SCOPE);
		
			PieceOfKnowledge pok = (PieceOfKnowledge) req.getEvent().getValue();
			/*
			 * getting the document URI using RDFSelector
			 */
			String docURI = getURIFromPoK(pok);
		
			if (docURI != null) {
				/*
				 * creating the document using the retrieve URI
				 */
				doc_evt = new Document();
				doc_evt.setUri(docURI);
				logger.info("receive an unloaded document with URI : "
						+ doc_evt.getUri());
			} else {
				// no URI found in pok
				try {
					logger
							.info("receive an unloaded document with unavailable URI (null), corresponding PoK :"
									+ ResourceUtil.saveToXMLString(pok));
				} catch (WebLabCheckedException e) {
					logger.debug("Unable to get PoK XML");
				}
			}
		} else {
			logger.info("Unsupported event :"+req.getEvent().getQName());
		}
		
		if (doc_evt != null) {
			/*
			 * Setting media unit in PortletSession
			 */
			req.getPortletSession().setAttribute(USER_MEDIA_UNIT, doc_evt,
					PortletSession.APPLICATION_SCOPE);
			
			req.getPortletSession().setAttribute(USER_RES_ANNOTS_LOADED, false, PortletSession.APPLICATION_SCOPE);
		}
		logger.info("process event "+req.getEvent().getQName()+"reponse time :"
				+ Long.toString(System.currentTimeMillis() - start));
	}

	@SuppressWarnings("unchecked")
	@Override
	public void serveResource(ResourceRequest request, ResourceResponse response)
			throws PortletException, IOException {
		/*
		 * long to get duration
		 */
		long start = System.currentTimeMillis();

		/*
		 * user current document
		 */
		MediaUnit unit = null;

		if (request.getPortletSession().getAttribute(USER_MEDIA_UNIT,
				PortletSession.APPLICATION_SCOPE) != null
				&& request.getPortletSession().getAttribute(
						CURRENT_RESSOURCE_IS_LOADED,
						PortletSession.APPLICATION_SCOPE) != null) {
			/*
			 * get document from session
			 */
			unit = (MediaUnit) request.getPortletSession().getAttribute(
					USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE);

			/*
			 * checking the document is load
			 */
			if ((Boolean) request.getPortletSession().getAttribute(
					CURRENT_RESSOURCE_IS_LOADED,
					PortletSession.APPLICATION_SCOPE) == false) {
				/*
				 * Document need to be loaded from repository
				 */
				String uri_to_load = unit.getUri();

				if (uri_to_load != null) {
					/*
					 * load a resource
					 */
					boolean loaded = false;
					logger.info("new resource to load : " + uri_to_load);

					/*
					 * getting Document from repository
					 */
					LoadResourceArgs args = new LoadResourceArgs();
					args.setResourceId(uri_to_load);
					Resource loadedResource = null;

					/*
					 * setting user repo url
					 */
					if (request.getPortletSession().getAttribute(
							USER_REPO_CONF, PortletSession.APPLICATION_SCOPE) == null) {
						RepoServiceConfigBean repo_conf = new RepoServiceConfigBean();
						repo_conf
								.setServiceURL(getInitParameter("repo_service_url"));
						request.getPortletSession().setAttribute(
								USER_REPO_CONF, repo_conf,
								PortletSession.APPLICATION_SCOPE);
					}
					URL repoURL = new URL(((RepoServiceConfigBean) request
							.getPortletSession().getAttribute(USER_REPO_CONF,
									PortletSession.APPLICATION_SCOPE))
							.getServiceURL());
					logger.info("use repository : "+repoURL.toString());
					try {
						/*
						 * call web service
						 */
						ResourceContainer repo;
						repo = getRepoService(repoURL);
						loadedResource = repo.loadResource(args).getResource();
						loaded = true;
					} catch (Exception e) {
						
						logger.error(e);
						request.setAttribute(ERROR,
								"Unable to get resource on repository service");
						loaded = false;
					}

					if (loaded && loadedResource instanceof MediaUnit) {
						unit = (MediaUnit) loadedResource;

						/*
						 * Setting media unit in PortletSession and
						 * CURRENT_RESOURCE_IS_LOADED
						 */
						request.getPortletSession().setAttribute(
								USER_MEDIA_UNIT, unit,
								PortletSession.APPLICATION_SCOPE);
						request.getPortletSession().setAttribute(
								CURRENT_RESSOURCE_IS_LOADED, true,
								PortletSession.APPLICATION_SCOPE);
					}
				}
			}

			/*
			 * resource loaded, checking for problem when loading
			 */

			if (request.getAttribute(ERROR) != null) {
				/*
				 * Dispatch view to JSP page
				 */
				response.setContentType("text/html");
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(
								getInitParameter("error_page_url"));
				rd.include(request, response);
			} else {
				/*
				 * no error during loading resource : continuing serving
				 * resource
				 */
				Statements mediaUnitWTMap = null;
				Document doc = (Document) request.getPortletSession().getAttribute(USER_MEDIA_UNIT, PortletSession.APPLICATION_SCOPE);
				
				if (request.getPortletSession().getAttribute(USER_RES_ANNOTS_LOADED, PortletSession.APPLICATION_SCOPE)!=null) {
					if (!(Boolean)request.getPortletSession().getAttribute(USER_RES_ANNOTS_LOADED, PortletSession.APPLICATION_SCOPE)) {
						
						/*
						 * getting rdf map from rs
						 */
						AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
						rdfSelector.limitToFirstLevelAnnotation(true);
						mediaUnitWTMap = rdfSelector.searchFor(doc, rdfProperties.getDisctinctAnnotationsNS());
						
						request.getPortletSession().setAttribute(USER_MEDIA_UNIT_WTMAP, mediaUnitWTMap, PortletSession.APPLICATION_SCOPE);
						request.getPortletSession().setAttribute(USER_RES_ANNOTS_LOADED, true, PortletSession.APPLICATION_SCOPE);
					}
				}
				
				
				if (request.getPortletSession().getAttribute(USER_MEDIA_UNIT_WTMAP, PortletSession.APPLICATION_SCOPE) != null ) {
					mediaUnitWTMap = (Statements) request.getPortletSession().getAttribute(USER_MEDIA_UNIT_WTMAP, PortletSession.APPLICATION_SCOPE);
					Map<String, List<String>> res_desc = new HashMap<String, List<String>>();
					
					for (WTriple statement : mediaUnitWTMap.get(doc.getUri()).getStatements()) {
						
						LinkedList<String> l = new LinkedList<String>();
						if (mediaUnitWTMap.get(doc.getUri()).getValue(statement.getPredicate()) instanceof List<?>) {
							l.addAll((List<String>)mediaUnitWTMap.get(doc.getUri()).getValue(statement.getPredicate()));
						} else {
							l.add((String) mediaUnitWTMap.get(doc.getUri()).getValue(statement.getPredicate()));
						}
						res_desc.put(statement.getPredicate(), l);
					}
					
					/*
					 * injecting meta-data
					 */
					request.setAttribute(META_MAP_DESC, res_desc);
					
					/*
					 * injecting resource uri
					 */
					request.setAttribute(USER_MEDIA_UNIT_URI, doc.getUri());
					
					/*
					 * injecting rdf bean
					 */
					request.setAttribute("rdfProperties", this.rdfProperties.getProperties());
				}
				
				
				/*
				 * getting description map from splitter
				 */
				logger.debug("resource map " + mediaUnitWTMap);

				/*
				 * getting meta configuration from session
				 */
				MetaConfBean res_conf = (MetaConfBean) request
						.getPortletSession().getAttribute(USER_META_CONF,
								PortletSession.APPLICATION_SCOPE);
				if (res_conf == null) {
					/*
					 * injecting meta config
					 */
					request.setAttribute(USER_META_CONF, DEFAULT_META_CONF_BEAN);
				} else {
					/*
					 * injecting meta config
					 */
					request.setAttribute(USER_META_CONF, res_conf);
				}
				
				/*
				 * dispatch to jsp
				 */
				response.setContentType("text/html");
				PortletRequestDispatcher rd = getPortletContext()
						.getRequestDispatcher(
								getInitParameter("view_metat_ajax_page_url"));
				rd.include(request, response);

			}
		}
				
		logger.info(
				"serveResource reponse time :"
						+ Long.toString(System.currentTimeMillis() - start));
	}

	/**
	 * Method to get Respository service from it URL
	 * 
	 * @param serviceURL
	 *            url of the service
	 * @return corresponding service
	 */

	private ResourceContainer getRepoService(URL serviceURL) {

		if (!this.repoMap.containsKey(serviceURL)) {
			/*
			 * web service doesn't exist, create it and add it to service map
			 */
			logger.info("Building repo at [" + serviceURL + "]");
			

			this.repoMap.put(serviceURL, ResourceContainerUtil
					.getResourceContainerService(WSDL_REPO_URL, serviceURL));
		}
		return this.repoMap.get(serviceURL);
	}

	/**
	 * Method to get URI from a receive PieceOfKnowledge using
	 * POK_PROPERTY_DOC_URI
	 * 
	 * @param pok
	 * @return URI if exist null otherwise
	 */
	private String getURIFromPoK(PieceOfKnowledge pok) {
		if (pok != null) {
			AdvancedSelector rdfSelector = RDFSelectorFactory.getSelector(true);
			Statements map = rdfSelector.searchFor(pok,
					POK_PROPERTY_DOC_URI);

			if (map != null && !map.entrySet().isEmpty()) {
				return (String) map.entrySet().iterator().next().getValue().getValue(POK_PROPERTY_DOC_URI);
			}
		}
		return null;
	}
}
