package org.ow2.weblab.portlet;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

import javax.portlet.ActionResponse;
import javax.portlet.EventResponse;
import javax.portlet.GenericPortlet;
import javax.portlet.PortletException;
import javax.xml.namespace.QName;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.model.Resource;

/**
 * The Class WebLabPortlet.
 */
public abstract class WebLabPortlet extends GenericPortlet {

	/** The reaction event map. */
	private transient Map<QName, QName> reactionEventMap = new HashMap<QName, QName>();

	/** The action event map. */
	private transient Map<QName, QName> actionEventMap = new HashMap<QName, QName>();

	/** The redirect event map. */
	private transient Map<QName, String> redirectEventMap = new HashMap<QName, String>();
	
	/** The Constant PORTLET_ACTION_NAMESPACE. */
	public static final String PORTLET_ACTION_NAMESPACE = "{http://weblab.ow2.org/portlet/action}";

	/** The Constant PORTLET_REACTION_NAMESPACE. */
	public static final String PORTLET_REACTION_NAMESPACE = "{http://weblab.ow2.org/portlet/reaction}";

	/** The Constant PORTLET_REDIRECT_NAMESPACE. */
	public static final String PORTLET_REDIRECT_NAMESPACE = "{http://weblab.ow2.org/portlet/redirection}";
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.portlet.GenericPortlet#init()
	 */
	@Override
	public void init() throws PortletException {
		initMappings();
	}
	
	/**
	 * Gets the reaction event map.
	 * 
	 * @return the reaction event map
	 */
	public Map<QName, QName> getReactionEventMap() {
		return this.reactionEventMap;
	}

	/**
	 * Gets the action event map.
	 * 
	 * @return the action event map
	 */
	public Map<QName, QName> getActionEventMap() {
		return this.actionEventMap;
	}

	/**
	 * Gets the redirect event map.
	 * 
	 * @return the redirect event map
	 */
	public Map<QName, String> getRedirectEventMap() {
		return this.redirectEventMap;
	}

	/**
	 * Gets the reaction associated to the QName of the event received (from
	 * file portlet.xml in your project).
	 * 
	 * @param specifiedQName
	 *            the specified q name
	 * 
	 * @return the reaction
	 */
	public QName getReaction(QName specifiedQName) {
		return this.reactionEventMap.get(specifiedQName);
	}

	/**
	 * Send event for action.
	 * 
	 * @param action
	 *            the action associated to a QName in the portlet.xml file
	 * @param resource
	 *            the resource to send in the event
	 * @param actionResponse
	 *            the response of the processAction
	 * 
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void sendEventForAction(String action, Resource resource, ActionResponse actionResponse) {
		if (this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action) !=null && ! this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action).equals("")) {
			QName qnOfAction = QName.valueOf(this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action));
			actionResponse.setEvent(qnOfAction, resource);
		} else {
			LogFactory.getLog(this.getClass()).info("Warning : You try to send an event for an undefined action ("+PORTLET_ACTION_NAMESPACE + action+"). Check your portlet.xml file.");
		}
	}
	
	
	/**
	 * Send event for action.
	 * 
	 * @param action
	 *            the action associated to a QName in the portlet.xml file
	 * @param resource
	 *            the resource to send in the event
	 * @param eventResponse
	 *            the response of the processEvent
	 * 
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void sendEventForAction(String action, Resource resource, EventResponse eventResponse) {
		if (this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action) !=null && ! this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action).equals("")) {
			QName qnOfAction = QName.valueOf(this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action));
			eventResponse.setEvent(qnOfAction, resource);
		} else {
			LogFactory.getLog(this.getClass()).info("Warning : You try to send an event for an undefined action ("+PORTLET_ACTION_NAMESPACE + action+"). Check your portlet.xml file.");
		}
	}
	
	

	/**
	 * Send event for action.
	 * 
	 * @param action
	 *            the action associated to a QName in the portlet.xml file
	 * @param resource
	 *            the resource to send in the event
	 * @param actionResponse
	 *            the response of the processAction
	 * 
	 *            WARNING : the method used for auto redirection is
	 *            sendRedirect, then it is not allow to setRenderParameter for
	 *            instance. For more information look for Exceptions that
	 *            sendRedirect can launch.
	 * 
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void sendEventForActionAndRedirect(String action, Resource resource, ActionResponse actionResponse)
			throws IOException {
		
		if (this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action) !=null && ! this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action).equals("")) {
			// send event
			QName qnOfAction = QName.valueOf(this.getPortletConfig().getInitParameter(PORTLET_ACTION_NAMESPACE + action));
			actionResponse.setEvent(qnOfAction, resource);
		

			// redirect
			String url = this.redirectEventMap.get(QName.valueOf(PORTLET_REDIRECT_NAMESPACE + action));
			if (url == null || url.equals("")) {
				LogFactory.getLog(this.getClass()).info(
						"Automatic redirection failed. Url null for the event(" + qnOfAction + ") from "
								+ actionResponse.getNamespace() + ".");
			} else {
				actionResponse.sendRedirect(this.redirectEventMap.get(QName.valueOf(PORTLET_REDIRECT_NAMESPACE + action)));
	
				LogFactory.getLog(this.getClass()).info(
						"Automatic redirection to url " + url + " after sending the event(" + qnOfAction + ") from "
								+ actionResponse.getNamespace() + ".");
			}
		} else {
			LogFactory.getLog(this.getClass()).info("Warning : You try to send an event for an undefined action ("+PORTLET_ACTION_NAMESPACE + action+"). Check your portlet.xml file.");
		}
	}

	
	/**
	 * Inits the mappings form the configuration file portlet.xml.
	 */
	private void initMappings() {
		Enumeration<String> initParam = this.getPortletConfig().getInitParameterNames();

		while (initParam.hasMoreElements()) {
			String param = initParam.nextElement();
			String value = this.getPortletConfig().getInitParameter(param);

			// reaction event mapping
			for (String s : value.split(" ")) {
				if (param.startsWith(PORTLET_REACTION_NAMESPACE)) {
					try {
						this.reactionEventMap.put(QName.valueOf(s), QName.valueOf(param));
					} catch (IllegalArgumentException e) {
						LogFactory.getLog(this.getClass()).warn(
										"Invalid reaction event mapping qname. Check the init parameters for event reaction mapping in portlet.xml.");
					}
				}
			}

			// action event mapping
			for (String s : value.split(" ")) {
				if (param.startsWith(PORTLET_ACTION_NAMESPACE)) {
					try {
						this.actionEventMap.put(QName.valueOf(s), QName.valueOf(param));
					} catch (IllegalArgumentException e) {
						LogFactory.getLog(this.getClass()).warn(
										"Invalid action event mapping qname. Check the init parameters for event action mapping in portlet.xml.");
					}
				}
			}

			// redirection mapping
			if (param.startsWith(PORTLET_REDIRECT_NAMESPACE)) {
				try {
					this.redirectEventMap.put(QName.valueOf(param), value);
				} catch (IllegalArgumentException e) {
					LogFactory.getLog(this.getClass()).warn(
									"Invalid redirect event mapping qname. Check the init parameters for event redirection mapping in portlet.xml.");
				}
			}
		}
	}

	/**
	 * Adds the redirection url to a specified action name.
	 * 
	 * @param qn
	 *            the qn (PORTLET_REDIRECT_NAMESPACE+action)
	 * @param url
	 *            the url (url of the page in the portal)
	 */
	public void addRedirection(QName qn, String url) {
		this.redirectEventMap.put(qn, url);
	}

	/**
	 * Modify redirection. If the QName already exists the url is overwrited,
	 * otherwise an error is logged
	 * 
	 * @param qn
	 *            the qn (PORTLET_REDIRECT_NAMESPACE+action)
	 * @param url
	 *            the url (url of the page in the portal)
	 */
	public void modifyRedirection(QName qn, String url) {
		if (this.redirectEventMap.containsKey(qn)) {
			this.redirectEventMap.put(qn, url);
		} else {
			LogFactory.getLog(this.getClass()).error(
					"Impossible to modify the redirect event map since the key (QName) has not been found");
		}
	}

	/**
	 * Delete redirection. If the QName already exists the url is deleted,
	 * otherwise an error is logged
	 * 
	 * @param qn
	 *            the qn
	 */
	public void deleteRedirection(QName qn) {
		if (this.redirectEventMap.containsKey(qn)) {
			this.redirectEventMap.remove(qn);
		} else {
			LogFactory.getLog(this.getClass()).error(
							"Impossible to delete the QName of the redirect event map since the key (QName) has not been found");
		}
	}

	/**
	 * Gets an accessible file from the Portlet context.
	 * 
	 * @param localPath
	 *            the local path
	 * 
	 * @return the file
	 * 
	 * @throws WebLabCheckedException
	 *             the web lab checked exception
	 */
	public File getFile(String localPath) throws WebLabCheckedException {
		File f = new File(getPortletContext().getRealPath(localPath));
		if (f.exists()) {
			return f;
		}
		throw new WebLabCheckedException("File " + localPath + " doesn't exist");
	}

	/**
	 * Gets the uRL.
	 * 
	 * @param localPath
	 *            the local path
	 * 
	 * @return the uRL
	 * 
	 * @throws WebLabCheckedException
	 *             the web lab checked exception
	 */
	public URL getURL(String localPath) throws WebLabCheckedException {
		File f = new File(getPortletContext().getRealPath(localPath));
		try {
			return f.toURI().toURL();
		} catch (MalformedURLException e) {
			throw new WebLabCheckedException("Invalid URL for File" + localPath);
		}
	}
}
