/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.blremover;

import java.util.List;
import java.util.regex.Pattern;

import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.weblab_project.core.model.ComposedUnit;
import org.weblab_project.core.model.Document;
import org.weblab_project.core.model.Resource;
import org.weblab_project.core.model.text.Text;
import org.weblab_project.core.util.ComposedUnitUtil;
import org.weblab_project.services.analyser.Analyser;
import org.weblab_project.services.analyser.ProcessException;
import org.weblab_project.services.analyser.types.ProcessArgs;
import org.weblab_project.services.analyser.types.ProcessReturn;
import org.weblab_project.services.exception.WebLabException;

/**
 * @author BA
 * @date 2009-07-03
 */
@WebService(endpointInterface = "org.weblab_project.services.analyser.Analyser")
public class BlankLineRemoverService implements Analyser {

	/**
	 * The <code>Log</code> to be used.
	 */
	private static Log LOG = LogFactory.getLog(BlankLineRemoverService.class);

	public BlankLineRemoverService() {
		super();
	}

	public ProcessReturn process(ProcessArgs args) throws ProcessException {
		BlankLineRemoverService.LOG.debug("Start of process method.");
		List<Text> textList = BlankLineRemoverService.checkArgs(args);
		Resource res = args.getResource();
		BlankLineRemoverService.LOG.info("Beginning process with MediaUnit: " + res.getUri() + ".");

		Pattern p = Pattern.compile("^\\s+$", Pattern.MULTILINE);
		
		for (Text text : textList)
		{
			String textContent = text.getContent();
			
			textContent = p.matcher(textContent).replaceAll("\n");
			textContent = textContent.replaceAll("[\\r\\n]{2,}|(?:\\r\\n){2,}$","\n\n");
			textContent = textContent.replace("\r\n", "\n");
			
			text.setContent(textContent);
		}
		
		ProcessReturn pr = new ProcessReturn();
		pr.setResource(res);

		BlankLineRemoverService.LOG.info("MediaUnit: " + res.getUri() + " successfully processed.");
		
		return pr;
	}

	/**
	 * Checks if <code>args</code> contains list of text sections and returns it.
	 * 
	 * @param args
	 *            The <code>ProcessArgs</code>
	 * @return The <code>list of text sections</code> in <code>args</code>.
	 * @throws ProcessException
	 *             If <code>args</code> is <code>null</code>, contains a
	 *             <code>Resource</code> that is <code>null</code> or not a
	 *             <code>MediaUnit</code>, contains a <code>MediaUnit</code>
	 *             that is not a <code>Document</code>.
	 */
	private static List<Text> checkArgs(final ProcessArgs args)
			throws ProcessException {
		if (args == null) {
			throw BlankLineRemoverService.createProcessEx("ProcessArgs was null.",
					"E1", null, null);
		}
		final Resource res = args.getResource();
		if (res == null) {
			throw BlankLineRemoverService.createProcessEx("Resource in ProcessArgs "
					+ "was null.", "E1", null, null);
		}
		if (!(res instanceof Document)) {
			throw BlankLineRemoverService.createProcessEx("Resource in ProcessArgs "
					+ "was not a Document, but a " + res.getClass().getName()
					+ ".", "E1", null, "URI of the buggy resource: "
					+ res.getUri() + ".");
		}
		
		return ComposedUnitUtil.getSelectedSubMediaUnits((ComposedUnit) res, Text.class);		
	}

	/**
	 * Creates a well formed {@link ProcessException} using parameters.
	 * 
	 * @param message
	 *            The error message to be added to the created
	 *            <code>ProcessException</code>.
	 * @param errorId
	 *            The error Id of the <code>WebLabException</code> to be added
	 *            to the created <code>ProcessException</code>.
	 * @param cause
	 *            The cause of the error to be added to the created
	 *            <code>ProcessException</code> if not <code>null</code>.
	 * @param details
	 *            A detailed error message to be logged if not <code>null</code>.
	 * @return The created <code>ProcessException</code>
	 */
	private static ProcessException createProcessEx(final String message,
			final String errorId, final Throwable cause, final String details) {
		ProcessException pe;
		WebLabException wle = new WebLabException();
		wle.setErrorId(errorId);
		if (errorId.equals("E0")) {
			wle.setErrorMessage("Unexpected error.");
		} else if (errorId.equals("E1")) {
			wle.setErrorMessage("Invalid parameter.");
		} else if (errorId.equals("E2")) {
			wle.setErrorMessage("Insufficient resources.");
		} else if (errorId.equals("E3")) {
			wle.setErrorMessage("Content not available.");
		} else if (errorId.equals("E4")) {
			wle.setErrorMessage("Access denied.");
		} else if (errorId.equals("E5")) {
			wle.setErrorMessage("Unsupported request.");
		} else {
			BlankLineRemoverService.LOG.warn("Unknown error ID: " + errorId);
			wle.setErrorMessage("Unknown error.");
		}
		if (cause == null) {
			BlankLineRemoverService.LOG.error(message);
			pe = new ProcessException(message, wle);
		} else {
			BlankLineRemoverService.LOG.error(message, cause);
			pe = new ProcessException(message, wle, cause);
		}
		if (details != null) {
			BlankLineRemoverService.LOG.error(details);
		}
		BlankLineRemoverService.LOG.debug("End of process method.");
		return pe;
	}
}
