package org.ow2.weblab.service.gate;

import gate.FeatureMap;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.LogFactory;
import org.weblab_project.core.comparator.SegmentComparator;
import org.weblab_project.core.factory.AnnotationFactory;
import org.weblab_project.core.factory.SegmentFactory;
import org.weblab_project.core.helper.PoKHelperExtended;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.model.Annotation;
import org.weblab_project.core.model.text.LinearSegment;
import org.weblab_project.core.model.text.Text;
import org.weblab_project.core.ontologies.WebLab;

/**
 * This class contains useful static methods for the gate-extraction project.
 * 
 * @author ymombrun
 * @date 2008-05-06
 */
public class GateHelper {

	private final static String GATE_TEMP_NS = WebLab.PROCESSING_PROPERTY_NAMESPACE + "temp/gate/";
	private final static String GATE_TEMP_NS_PREFIX = "tempGate";
	private final static Map<String, String> EMPTY_MAP = Collections.<String, String> emptyMap();

	/**
	 * The URI of the predicate for types.
	 * 
	 * Annotation are created on a temporary namespace. The reason is that Gate let you to write you own plugins, and rules, enabling the creation a your own Types.
	 * Since this service shall be working in various applications, having various business ontologies, the annotation format is a temporary one.
	 * Best practice is to write a service, that will be called after this to do the mapping between those temporary annotations and good annotation using the specified ontology.
	 */
	public final static String GATE_TEMP_ANNOTATION_TYPE = GATE_TEMP_NS + "type";

	/**
	 * The URI of the predicate for meta data that refines types.
	 */
	public final static String GATE_TEMP_ANNOTATION_META = GATE_TEMP_NS + "meta";

	/**
	 * Annotate text with each annotation in annotation set.
	 * At the end, sorts the segments list to ease further process.
	 * 
	 * @param text
	 *            The WebLab Text to be annotated
	 * @param annots
	 *            The Gate annotation set to be used to annotate text
	 * @param generatorURI
	 *            The URI of the service. This will be used to annotate the annotation with a produce from statement.
	 */
	public static void linkGateAnnotsToText(Text text, final gate.AnnotationSet annots, final String generatorURI) {
		Map<String, Annotation> annotated = new HashMap<String, Annotation>();
		for (final gate.Annotation annot : annots) {
			GateHelper.linkGateAnnotToText(text, annot, generatorURI, annotated);
		}
		Collections.sort(text.getSegment(), new SegmentComparator());
	}

	/**
	 * Creates a <code>LinearSegment</code> at the same position that the <code>gate.Annotation</code>.
	 * Create an <code>Annotation</code> on this <code>Segment</code>.
	 * If the same segment is already annotated by this service (contained by the <code>Map</code>) the existing <code>Annotation</code> is used.
	 * 
	 * Add to this <code>Annotation</code> the statements from <code>gate.Annotation</code>.
	 * 
	 * @param text
	 *            The text section to process
	 * @param annotGate
	 *            An annotation in gate format
	 * @param generatorURI
	 *            The URI of the generator of the annotation to be used to annotated the created annotation. If <code>null</code> or empty, the annotation will not be annotated.
	 * @param annotated
	 *            The <code>Map</code> of previously annotated by this service <code>Segment</code>, enabling to not create various <code>Annotation</code>s for the same position in <code>text</code>.
	 */
	private static void linkGateAnnotToText(Text text, final gate.Annotation annotGate, final String generatorURI, Map<String, Annotation> annotated) {
		LinearSegment segment = SegmentFactory.createAndLinkLinearSegment(text, annotGate.getStartNode().getOffset().intValue(), annotGate.getEndNode().getOffset().intValue());

		final PoKHelperExtended annotationHelper;
		if (annotated.containsKey(segment.getUri())) {
			final Annotation annot = annotated.get(segment.getUri());
			annotationHelper = RDFHelperFactory.getPoKHelperExtended(annot);
		} else {
			final Annotation annot = AnnotationFactory.createAndLinkAnnotation(segment);
			annotated.put(segment.getUri(), annot);
			annotationHelper = RDFHelperFactory.getPoKHelperExtended(annot);
			annotationHelper.setNSPrefix(GATE_TEMP_NS_PREFIX, GATE_TEMP_NS);
			if (generatorURI != null) {
				Annotation generatedAnnot = AnnotationFactory.createAndLinkAnnotation(annot);
				PoKHelperExtended pokHelper2 = RDFHelperFactory.getPoKHelperExtended(generatedAnnot);
				pokHelper2.setNSPrefix("wlp", WebLab.PROCESSING_PROPERTY_NAMESPACE);
				pokHelper2.createResStat(annot.getUri(), WebLab.IS_PRODUCED_BY, generatorURI);
			}
		}
		annotationHelper.setAutoCommitMode(false);
		GateHelper.reifyMetas(annotGate, segment.getUri(), annotationHelper);
		annotationHelper.commit();
	}

	/**
	 * Reads the content annotGate featureMap and annotate as reified statement these properties. It also annotate startNode, endNode and id of the Gate Annotation.
	 * 
	 * @param annotGate
	 *            The annotation to extract features
	 * @param subject
	 *            The subject of the statement
	 * @param pokHE
	 *            the extended pok helper to be used to annotate
	 */
	private static void reifyMetas(final gate.Annotation annotGate, final String subject, final PoKHelperExtended pokHE) {
		final String reifURI = "weblab:gateAnnot/" + annotGate.getType() + "/" + System.nanoTime() + "/" + annotGate.hashCode();
		final FeatureMap featureMap = annotGate.getFeatures();
		if (featureMap != null && !featureMap.isEmpty()) {
			for (final Object key : featureMap.keySet()) {
				if (key instanceof String) {
					String featKey = ((String) key).trim();
					final Object featureValue = featureMap.get(featKey);
					if (featureValue != null) {
						GateHelper.reifyLiteralMeta(subject, annotGate.getType(), featKey + "=" + featureValue.toString(), pokHE, reifURI);
					}

				} else {
					LogFactory.getLog(GateHelper.class).warn("Unable to create feature from key '" + key + "' on gate annotation type '" + annotGate.getType() + "'.");
				}
			}
		}
		GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateStartNode=" + annotGate.getStartNode().getId().toString(), pokHE, reifURI);
		GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateEndNode=" + annotGate.getEndNode().getId().toString(), pokHE, reifURI);
		GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateAnnotId=" + annotGate.getId().toString(), pokHE, reifURI);
	}

	/**
	 * @param subject
	 *            The subject of the original statement
	 * @param object
	 *            The object of the original statement
	 * @param reifiedLiteral
	 *            The reified literal value
	 * @param pokHE
	 *            The pokHE to be used to annotate
	 * @param reifURI
	 *            The URI of the reified statement to be created.
	 */
	private static void reifyLiteralMeta(final String subject, final String object, final String reifiedLiteral, PoKHelperExtended pokHE, final String reifURI) {
		Map<String, String> properFeatures = new HashMap<String, String>(1);
		properFeatures.put(GATE_TEMP_ANNOTATION_META, reifiedLiteral);
		pokHE.createLitStatReif(subject, GATE_TEMP_ANNOTATION_TYPE, object, properFeatures, EMPTY_MAP, EMPTY_MAP, reifURI);
	}

}