package org.ow2.weblab.service.gate;

import gate.FeatureMap;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.weblab_project.core.comparator.SegmentComparator;
import org.weblab_project.core.factory.AnnotationFactory;
import org.weblab_project.core.factory.SegmentFactory;
import org.weblab_project.core.helper.PoKHelperExtended;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.model.Annotation;
import org.weblab_project.core.model.text.LinearSegment;
import org.weblab_project.core.model.text.Text;
import org.weblab_project.core.ontologies.WebLab;
import org.weblab_project.core.properties.PropertiesLoader;


/**
 * This class contains useful static methods for the gate-extraction project.
 * 
 * @author ymombrun
 * @date 2008-05-06
 */
public class GateHelper {

	private final static String GATE_TEMP_NS = WebLab.PROCESSING_PROPERTY_NAMESPACE + "temp/gate/";

	private final static String GATE_TEMP_NS_PREFIX = "tempGate";

	private final static Map<String, String> EMPTY_MAP = Collections.<String, String> emptyMap();

	private final static String PROPERTIES_FILE_NAME = "gate.properties";

	private final static String EXCLUSION_PROPERTY = "exclude";

	private final static String FEATURE_EXCLUSION_PROPERTY = "exludeFeatures";

	private final static String ADD_START_NODE_ID_PROPERTY = "addStartNodeId";

	private final static String ADD_END_NODE_ID_PROPERTY = "addEndNodeId";

	private final static String ADD_ANNOT_ID_PROPERTY = "addAnnotId";

	private final static String SERVICE_URI_PROPERTY = "serviceURI";

	private static Set<String> EXCLUDED_ANNOTATIONS = new HashSet<String>();
	private static Set<String> EXCLUDED_FEATURES = new HashSet<String>();

	private static boolean ADD_START_NODE_ID;
	private static boolean ADD_END_NODE_ID;
	private static boolean ADD_ANNOT_ID;
	private static String SERVICE_URI;


	/**
	 * The URI of the predicate for types.
	 * 
	 * Annotation are created on a temporary namespace. The reason is that Gate let you to write you own plugins, and rules, enabling the creation a your own Types.
	 * Since this service shall be working in various applications, having various business ontologies, the annotation format is a temporary one.
	 * Best practice is to write a service, that will be called after this to do the mapping between those temporary annotations and good annotation using the specified ontology.
	 */
	public final static String GATE_TEMP_ANNOTATION_TYPE = GATE_TEMP_NS + "type";

	/**
	 * The URI of the predicate for meta data that refines types.
	 */
	public final static String GATE_TEMP_ANNOTATION_META = GATE_TEMP_NS + "meta";

	/**
	 * Annotate text with each annotation in annotation set.
	 * At the end, sorts the segments list to ease further process.
	 * 
	 * @param text
	 *            The WebLab Text to be annotated
	 * @param annots
	 *            The Gate annotation set to be used to annotate text
	 */
	public static void linkGateAnnotsToText(Text text, final gate.AnnotationSet annots) {
		Map<String, Annotation> annotated = new HashMap<String, Annotation>();
		for (final gate.Annotation annot : annots) {
			if (!EXCLUDED_ANNOTATIONS.contains(annot.getType())) {
				GateHelper.linkGateAnnotToText(text, annot, annotated);
			}
		}
		Collections.sort(text.getSegment(), new SegmentComparator());
	}

	/**
	 * Creates a <code>LinearSegment</code> at the same position that the <code>gate.Annotation</code>.
	 * Create an <code>Annotation</code> on this <code>Segment</code>.
	 * If the same segment is already annotated by this service (contained by the <code>Map</code>) the existing <code>Annotation</code> is used.
	 * 
	 * Add to this <code>Annotation</code> the statements from <code>gate.Annotation</code>.
	 * 
	 * @param text
	 *            The text section to process
	 * @param annotGate
	 *            An annotation in gate format
	 * @param annotated
	 *            The <code>Map</code> of previously annotated by this service <code>Segment</code>, enabling to not create various <code>Annotation</code>s for the same position in <code>text</code>.
	 */
	private static void linkGateAnnotToText(Text text, final gate.Annotation annotGate, Map<String, Annotation> annotated) {
		LinearSegment segment = SegmentFactory.createAndLinkLinearSegment(text, annotGate.getStartNode().getOffset().intValue(), annotGate.getEndNode().getOffset().intValue());

		final PoKHelperExtended pokHelper;
		if (annotated.containsKey(segment.getUri())) {
			final Annotation annot = annotated.get(segment.getUri());
			pokHelper = RDFHelperFactory.getPoKHelperExtended(annot);
			pokHelper.setAutoCommitMode(false);
		} else {
			final Annotation annot = AnnotationFactory.createAndLinkAnnotation(segment);
			annotated.put(segment.getUri(), annot);
			pokHelper = RDFHelperFactory.getPoKHelperExtended(annot);
			pokHelper.setAutoCommitMode(false);
			pokHelper.setNSPrefix(GATE_TEMP_NS_PREFIX, GATE_TEMP_NS);
			if (SERVICE_URI != null) {
				pokHelper.setNSPrefix("wlp", WebLab.PROCESSING_PROPERTY_NAMESPACE);
				pokHelper.createResStat(annot.getUri(), WebLab.IS_PRODUCED_BY, SERVICE_URI);
			}
		}
		GateHelper.reifyMetas(annotGate, segment.getUri(), pokHelper);
		pokHelper.commit();
	}

	/**
	 * Reads the content annotGate featureMap and annotate as reified statement these properties. It also annotate startNode, endNode and id of the Gate Annotation.
	 * 
	 * @param annotGate
	 *            The annotation to extract features
	 * @param subject
	 *            The subject of the statement
	 * @param pokHE
	 *            the extended pok helper to be used to annotate
	 */
	private static void reifyMetas(final gate.Annotation annotGate, final String subject, final PoKHelperExtended pokHE) {
		final String reifURI = "weblab:gateAnnot/" + annotGate.getType() + "/" + System.nanoTime() + "/" + annotGate.hashCode();
		boolean reified = false;
		final FeatureMap featureMap = annotGate.getFeatures();
		if (featureMap != null && !featureMap.isEmpty()) {
			for (final Object key : featureMap.keySet()) {
				if (key instanceof String) {
					String featKey = ((String) key).trim();
					if (!EXCLUDED_FEATURES.contains(featKey)) {
						final Object featureValue = featureMap.get(featKey);
						if (featureValue != null) {
							GateHelper.reifyLiteralMeta(subject, annotGate.getType(), featKey + "=" + featureValue.toString(), pokHE, reifURI);
							reified = true;
						}
					}
				} else {
					LogFactory.getLog(GateHelper.class).warn("Unable to create feature from key '" + key + "' on gate annotation type '" + annotGate.getType() + "'.");
				}
			}
		}

		if (ADD_START_NODE_ID) {
			GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateStartNode=" + annotGate.getStartNode().getId().toString(), pokHE, reifURI);
			reified = true;
		}
		if (ADD_END_NODE_ID) {
			GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateEndNode=" + annotGate.getEndNode().getId().toString(), pokHE, reifURI);
			reified = true;
		}
		if (ADD_ANNOT_ID) {
			GateHelper.reifyLiteralMeta(subject, annotGate.getType(), "gateAnnotId=" + annotGate.getId().toString(), pokHE, reifURI);
			reified = true;
		}
		
		if (!reified) {
			pokHE.createLitStat(subject, GATE_TEMP_ANNOTATION_TYPE, annotGate.getType());
		}
	}

	/**
	 * @param subject
	 *            The subject of the original statement
	 * @param object
	 *            The object of the original statement
	 * @param reifiedLiteral
	 *            The reified literal value
	 * @param pokHE
	 *            The pokHE to be used to annotate
	 * @param reifURI
	 *            The URI of the reified statement to be created.
	 */
	private static void reifyLiteralMeta(final String subject, final String object, final String reifiedLiteral, PoKHelperExtended pokHE, final String reifURI) {
		Map<String, String> properFeatures = new HashMap<String, String>(1);
		properFeatures.put(GATE_TEMP_ANNOTATION_META, reifiedLiteral);
		pokHE.createLitStatReif(subject, GATE_TEMP_ANNOTATION_TYPE, object, properFeatures, EMPTY_MAP, EMPTY_MAP, reifURI);
	}



	/**
	 * Use the file PROPERTIES_FILE_NAME (gate.properties) to set:
	 * <ul>
	 * <li>Which annotation types to exclude.</li>
	 * <li>Whether or not to add the start node id.</li>
	 * <li>Whether or not to add the end node id.</li>
	 * <li>Whether or not to add the annot id.</li>
	 * <li>Which service URI to use with isProducedBy statements (if not defined, not statement will be added).</li>
	 * </ul>
	 */
	public static void init() {
		Map<String, String> props = PropertiesLoader.loadProperties(PROPERTIES_FILE_NAME);

		final String excludedString = props.get(EXCLUSION_PROPERTY);
		final String excludedFeaturesString = props.get(FEATURE_EXCLUSION_PROPERTY);
		final String addStartNodeIdString = props.get(ADD_START_NODE_ID_PROPERTY);
		final String addEndNodeIdString = props.get(ADD_END_NODE_ID_PROPERTY);
		final String addAnnotIdString = props.get(ADD_ANNOT_ID_PROPERTY);

		SERVICE_URI = props.get(SERVICE_URI_PROPERTY);

		if (excludedString != null && !excludedString.isEmpty()) {
			EXCLUDED_ANNOTATIONS.addAll(Arrays.asList(excludedString.split(";")));
		} else {
			LogFactory.getLog(GateHelper.class).warn("No " + EXCLUSION_PROPERTY + " property found in " + PROPERTIES_FILE_NAME + " file; nothing will be skipped.");
		}

		if (excludedFeaturesString != null && !excludedFeaturesString.isEmpty()) {
			EXCLUDED_FEATURES.addAll(Arrays.asList(excludedFeaturesString.split(";")));
		} else {
			LogFactory.getLog(GateHelper.class).warn("No " + FEATURE_EXCLUSION_PROPERTY + " property found in " + PROPERTIES_FILE_NAME + " file; nothing will be skipped.");
		}

		if (addAnnotIdString != null) {
			ADD_ANNOT_ID = Boolean.parseBoolean(addAnnotIdString);
		}
		if (addStartNodeIdString != null) {
			ADD_START_NODE_ID = Boolean.parseBoolean(addStartNodeIdString);
		}
		if (addEndNodeIdString != null) {
			ADD_END_NODE_ID = Boolean.parseBoolean(addEndNodeIdString);
		}

	}

}