/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.service.gate;

import java.io.File;
import java.util.List;

import javax.jws.WebService;

import org.apache.commons.logging.LogFactory;
import org.weblab_project.core.helper.PoKHelper;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.ontologies.WebLab;
import org.weblab_project.services.configurable.Configurable;
import org.weblab_project.services.configurable.ConfigureException;
import org.weblab_project.services.configurable.ResetConfigurationException;
import org.weblab_project.services.configurable.types.ConfigureArgs;
import org.weblab_project.services.configurable.types.ConfigureReturn;
import org.weblab_project.services.configurable.types.ResetConfigurationArgs;
import org.weblab_project.services.configurable.types.ResetConfigurationReturn;


/**
 * The configurable part of the Gate service.
 * It enables to defines the path to the gapp file that has to be used to create a Gate pipeline for the given
 * usageContext.
 * To do this you have to call the configure method using a PieceOfKnowledge that contains the following RDF statement:
 * <ul>
 * <li>Subject: the URI of the service (default value is DEFAULT_SERVICE_URI)</li>
 * <li>Predicate: GAPP_FILE_PATH_PROPERTY</li>
 * <li>Object: A Literal String containing the (absolute) path to the gapp file to be used.</li>
 * </ul>
 * 
 * @author ymombrun
 * @date 2010-08-31
 */
@WebService(endpointInterface = "org.weblab_project.services.configurable.Configurable")
public class GateConfiguratorService implements Configurable {

	/**
	 * The default service URI to be used.
	 */
	public static String DEFAULT_SERVICE_URI = "http://weblab-project.org/webservices/gateservice";

	/**
	 * The predicate of the configuration property
	 */
	public static final String GAPP_FILE_PATH_PROPERTY = WebLab.PROCESSING_PROPERTY_NAMESPACE + "gate" + "/gappFilePath";

	/**
	 * The service URI to be used
	 */
	private String serviceURI;

	/**
	 * Default constructor, using DEFAULT_SERVICE_URI as serviceURI
	 */
	public GateConfiguratorService() {
		this(DEFAULT_SERVICE_URI);
	}

	/**
	 * @param serviceURI
	 *            The serviceURI to be set
	 */
	public GateConfiguratorService(final String serviceURI) {
		super();
		this.serviceURI = serviceURI;
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.weblab_project.services.configurable.Configurable#configure(org.weblab_project.services.configurable.types
	 * .ConfigureArgs)
	 */
	@Override
	public ConfigureReturn configure(final ConfigureArgs args) throws ConfigureException {
		LogFactory.getLog(this.getClass()).info("Configure method called.");

		final PoKHelper helper = this.checkArgs(args);

		final List<String> gappFilePaths = helper.getLitsOnPredSubj(this.serviceURI, GAPP_FILE_PATH_PROPERTY);

		if (gappFilePaths.isEmpty()) {
			throw new ConfigureException("PieceOfKnowledge of used for configuration does not contains required statement: (S,P,O) = ('"
					+ this.serviceURI + "', '" + GAPP_FILE_PATH_PROPERTY + "', PathToGappFile)", WebLabExceptions.E1.wle);
		}

		final String gappFilePath = gappFilePaths.get(0);
		if (gappFilePaths.size() > 1) {
			LogFactory.getLog(this.getClass()).warn(
					"More than one statement (S,P,O) = ('" + this.serviceURI + "', '" + GAPP_FILE_PATH_PROPERTY
							+ "', PathToGappFile) found in configuration pok.");
		}

		this.checkFile(gappFilePath);

		Configuration.getInstance().setGateApplicationStateFileName(args.getUsageContext().getUri(), gappFilePath);
		LogFactory.getLog(this.getClass()).info("GappFile affected : " + gappFilePath);

		return new ConfigureReturn();
	}


	/**
	 * Checks if the gappFilePath is readable.
	 * 
	 * @param gappFilePath
	 *            The path to the gapp file to be used.
	 * @throws ConfigureException
	 *             If the file either: does not exists, is not a file, cannot be read.
	 */
	private void checkFile(final String gappFilePath) throws ConfigureException {
		final File file = new File(gappFilePath);
		if (!file.exists()) {
			throw new ConfigureException("GappFile '" + file.getAbsolutePath() + "' does not exists.", WebLabExceptions.E2.wle);
		}
		if (!file.isFile()) {
			throw new ConfigureException("GappFile '" + file.getAbsolutePath() + "' is not a file.", WebLabExceptions.E2.wle);
		}
		if (!file.canRead()) {
			throw new ConfigureException("GappFile '" + file.getAbsolutePath() + "' cannot be read.", WebLabExceptions.E2.wle);
		}
	}


	/**
	 * Checks the configureArgs and creates a pokhleper on the configuration pok.
	 * 
	 * @param args
	 *            The configure args of the configure method
	 * @return A PoKHelper loaded on the configuration pok
	 * @throws ConfigureException
	 *             If one of args, usageContext, configuration or usageContext URI is null.
	 */
	private PoKHelper checkArgs(final ConfigureArgs args) throws ConfigureException {
		if (args == null) {
			throw new ConfigureException("ConfigureArgs was null", WebLabExceptions.E1.wle);
		}
		if (args.getUsageContext() == null) {
			throw new ConfigureException("UsageContext was null", WebLabExceptions.E1.wle);
		}
		if (args.getConfiguration() == null) {
			throw new ConfigureException("Configuration was null", WebLabExceptions.E1.wle);
		}
		if (args.getUsageContext().getUri() == null) {
			throw new ConfigureException("URI of UsageContext was null", WebLabExceptions.E1.wle);
		}
		return RDFHelperFactory.getPoKHelper(args.getConfiguration());
	}

	/*
	 * (non-Javadoc)
	 * @see
	 * org.weblab_project.services.configurable.Configurable#resetConfiguration(org.weblab_project.services.configurable
	 * .types.ResetConfigurationArgs)
	 */
	@Override
	public ResetConfigurationReturn resetConfiguration(final ResetConfigurationArgs args) throws ResetConfigurationException {
		LogFactory.getLog(this.getClass()).info("ResetConfiguration method called.");

		if (args == null || args.getUsageContext() == null || args.getUsageContext().getUri() == null) {
			throw new ResetConfigurationException(
					"ResetConfigurationArgs was invalid (either it self, usageContext, it's uri or configuration was null",
					WebLabExceptions.E1.wle);
		}

		Configuration.getInstance().resetConfiguration(args.getUsageContext().getUri());

		return new ResetConfigurationReturn();
	}
}
