/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian SAS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.service.gate;

import gate.CorpusController;
import gate.Factory;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.LogFactory;


/**
 * This class is a singleton to be used to let the configurable and the analyser service communicate.
 * It holds the mapping between usageContext and gapp file path, and also usageContext and corpuscontroller.
 * 
 * @author ymombrun
 * @date 2010-08-31
 * @todo Add a pool handler to have more than one Controller per usageContext. (Since the process is synchronized on each Controller instance).
 */
public class Configuration {

	/**
	 * The singleton
	 */
	private static Configuration SINGLETON;

	/**
	 * The map of gapp file path by usageContext
	 */
	final private Map<String, String> usContextForGappFilePath = Collections.synchronizedMap(new HashMap<String, String>());

	/**
	 * The map of controller per gapp file path
	 */
	final private Map<String, CorpusController> gappFilePathForController = Collections.synchronizedMap(new HashMap<String, CorpusController>());


	/**
	 * The single constructor that shall not be call twice.
	 */
	private Configuration() {
		super();
	}

	/**
	 * @return The singleton instance of Configuration
	 */
	public static synchronized Configuration getInstance() {
		if (SINGLETON == null) {
			SINGLETON = new Configuration();
		}
		return SINGLETON;
	}


	/**
	 * @param usageContext
	 *            The usageContext to be used as key. May be empty but not null.
	 * @param gappFilePath
	 *            The gappFile to be mapped with the usageContext. Shall not be null.
	 */
	public synchronized void configure(final String usageContext, final String gappFilePath) {
		if (this.isControllerInitialised(usageContext)) {
			this.resetConfiguration(usageContext);
		}
		this.usContextForGappFilePath.put(usageContext, gappFilePath);
	}


	/**
	 * Removes every mappings in this object ; moreover it deletes the singleton.
	 */
	public synchronized void resetConfiguration() {
		for (final CorpusController controller : this.gappFilePathForController.values()) {
			Factory.deleteResource(controller);
		}
		this.gappFilePathForController.clear();
		this.usContextForGappFilePath.clear();
		LogFactory.getLog(this.getClass()).info("Every configuration were removed.");
		SINGLETON = null;
	}


	/**
	 * @param usageContext
	 *            The usageContext to be used as key. May be empty but not null.
	 * @return If the map contains an entry for this value
	 */
	public synchronized boolean isConfigured(final String usageContext) {
		return this.usContextForGappFilePath.containsKey(usageContext);
	}


	/**
	 * @param gappFilePath
	 *            The gappFilePath to be used as key.
	 * @return true, if a controller (may be the default one not already configured usageContext) has been initialised.
	 */
	public synchronized boolean isControllerInitialised(final String gappFilePath) {
		return this.gappFilePathForController.containsKey(gappFilePath);
	}


	/**
	 *  @param gappFilePath
	 *            The gappFilePath to be used as key.
	 * @return The controller that has been instanced for this gappFilePath.
	 */
	public synchronized CorpusController getController(final String gappFilePath) {
		return this.gappFilePathForController.get(gappFilePath);
	}


	/**
	 * @param usageContext
	 *            The usageContext to be used as key. May be empty but not null.
	 * @return The gapp file path
	 */
	public synchronized String getGappFilePathForUsageContext(final String usageContext) {
		return this.usContextForGappFilePath.get(usageContext);
	}


	/**
	 * @param usageContext
	 *            The usageContext to be used as key. May be empty but not null.
	 * @param filePath
	 *            The gapp file path
	 */
	public synchronized void setGappFileForUsageContext(final String usageContext, final String filePath) {
		this.usContextForGappFilePath.put(usageContext, filePath);
	}


	/**
	 * @param usageContext
	 *            The usageContext to be used as key. May be empty but not null.
	 * @param pipeline
	 *            The controller
	 */
	public synchronized void setControllerForUsageContextAndGapp(final String usageContext, final CorpusController pipeline) {
		final String gappFile = this.usContextForGappFilePath.get(usageContext);
		this.gappFilePathForController.put(gappFile, pipeline);
	}


	/**
	 * Remove entries in the two map that where associated with this usageContext.
	 * 
	 * @param usageContext
	 *            The usageContext that shall not be null
	 */
	public synchronized void resetConfiguration(final String usageContext) {
		final String gappFile = this.usContextForGappFilePath.remove(usageContext);

		if (gappFile == null) {
			LogFactory.getLog(this.getClass()).warn("Unable to resetConfiguration on usageContext '" + usageContext + "' since it has not been configured.");
		} else if (this.isControllerInitialised(usageContext) && (!this.usContextForGappFilePath.containsValue(gappFile))) {
			// Only remove the controller if it is initialised and if it is not used in another configuration.
			Factory.deleteResource(this.gappFilePathForController.get(gappFile));
		} else {
			LogFactory.getLog(this.getClass()).warn("Nothing to do since the controller has not been initialised.");
		}

		LogFactory.getLog(this.getClass()).info("Configuration for usageContext '" + usageContext + "' removed.");
	}


}
