/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2010 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.service.gate;

import java.io.File;
import java.util.List;

import javax.jws.WebService;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.ontologies.WebLab;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.RDFHelperFactory;
import org.ow2.weblab.core.services.Configurable;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.configurable.ConfigureArgs;
import org.ow2.weblab.core.services.configurable.ConfigureReturn;
import org.ow2.weblab.core.services.configurable.ResetConfigurationArgs;
import org.ow2.weblab.core.services.configurable.ResetConfigurationReturn;


/**
 * The configurable part of the Gate service.
 * It enables to defines the path to the gapp file that has to be used to create a Gate pipeline for the given
 * usageContext.
 * To do this you have to call the configure method using a PieceOfKnowledge that contains the following RDF statement:
 * <ul>
 * <li>Subject: the URI of the service (default value is DEFAULT_SERVICE_URI)</li>
 * <li>Predicate: GAPP_FILE_PATH_PROPERTY</li>
 * <li>Object: A Literal String containing the (absolute) path to the gapp file to be used.</li>
 * </ul>
 * 
 * @author ymombrun
 * @date 2010-08-31
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.Configurable")
public class GateConfiguratorService implements Configurable {

	/**
	 * The default service URI to be used.
	 */
	public static String DEFAULT_SERVICE_URI = "http://weblab.ow2.org/webservices/gateservice";

	/**
	 * The predicate of the configuration property
	 */
	public static final String GAPP_FILE_PATH_PROPERTY = WebLab.PROCESSING_PROPERTY_NAMESPACE + "gate" + "/gappFilePath";

	/**
	 * The service URI to be used
	 */
	private String serviceURI;

	/**
	 * Default constructor, using DEFAULT_SERVICE_URI as serviceURI
	 */
	public GateConfiguratorService() {
		this(DEFAULT_SERVICE_URI);
	}

	/**
	 * @param serviceURI
	 *            The serviceURI to be set
	 */
	public GateConfiguratorService(final String serviceURI) {
		super();
		this.serviceURI = serviceURI;
	}


	@Override
	public ConfigureReturn configure(final ConfigureArgs args) throws InvalidParameterException {
		LogFactory.getLog(this.getClass()).info("Configure method called.");

		final PoKHelper helper = this.checkArgs(args);

		final List<String> gappFilePaths = helper.getLitsOnPredSubj(this.serviceURI, GAPP_FILE_PATH_PROPERTY);

		if (gappFilePaths.isEmpty()) {
			final String exceptionMessage="PieceOfKnowledge of used for configuration does not contains required statement: (S,P,O) = ('"
				+ this.serviceURI + "', '" + GAPP_FILE_PATH_PROPERTY + "', PathToGappFile)";
			throw new InvalidParameterException(exceptionMessage, exceptionMessage);
		}

		final String gappFilePath = gappFilePaths.get(0);
		if (gappFilePaths.size() > 1) {
			LogFactory.getLog(this.getClass()).warn(
					"More than one statement (S,P,O) = ('" + this.serviceURI + "', '" + GAPP_FILE_PATH_PROPERTY
							+ "', PathToGappFile) found in configuration pok.");
		}

		this.checkFile(gappFilePath);

		Configuration.getInstance().setGateApplicationStateFileName(args.getUsageContext(), gappFilePath);
		LogFactory.getLog(this.getClass()).info("GappFile affected : " + gappFilePath);

		return new ConfigureReturn();
	}


	/**
	 * Checks if the gappFilePath is readable.
	 * 
	 * @param gappFilePath
	 *            The path to the gapp file to be used.
	 * @throws ConfigureException
	 *             If the file either: does not exists, is not a file, cannot be read.
	 */
	private void checkFile(final String gappFilePath) throws InvalidParameterException {
		final File file = new File(gappFilePath);
		if (!file.exists()) {
			throw new InvalidParameterException("GappFile '" + file.getAbsolutePath() + "' does not exists.", "GappFile '" + file.getAbsolutePath() + "' does not exists.");
		}
		if (!file.isFile()) {
			throw new InvalidParameterException("GappFile '" + file.getAbsolutePath() + "' is not a file.", "GappFile '" + file.getAbsolutePath() + "' does not exists.");
		}
		if (!file.canRead()) {
			throw new InvalidParameterException("GappFile '" + file.getAbsolutePath() + "' cannot be read.", "GappFile '" + file.getAbsolutePath() + "' does not exists.");
		}
	}


	/**
	 * Checks the configureArgs and creates a pokhleper on the configuration pok.
	 * 
	 * @param args
	 *            The configure args of the configure method
	 * @return A PoKHelper loaded on the configuration pok
	 * @throws ConfigureException
	 *             If one of args, usageContext, configuration or usageContext URI is null.
	 */
	private PoKHelper checkArgs(final ConfigureArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("ConfigureArgs was null", "ConfigureArgs was null");
		}
		if (args.getUsageContext() == null) {
			throw new InvalidParameterException("UsageContext was null", "UsageContext was null");
		}
		if (args.getConfiguration() == null) {
			throw new InvalidParameterException("Configuration was null", "Configuration was null");
		}
		
		return RDFHelperFactory.getPoKHelper(args.getConfiguration());
	}


	@Override
	public ResetConfigurationReturn resetConfiguration(final ResetConfigurationArgs args) throws InvalidParameterException {
		LogFactory.getLog(this.getClass()).info("ResetConfiguration method called.");

		if (args == null || args.getUsageContext() == null || args.getUsageContext() == null) {
			throw new InvalidParameterException(
					"ResetConfigurationArgs was invalid (either it self, usageContext, it's uri or configuration was null",
					"ResetConfigurationArgs was invalid (either it self, usageContext, it's uri or configuration was null");
		}

		Configuration.getInstance().resetConfiguration(args.getUsageContext());

		return new ResetConfigurationReturn();
	}
}
