package org.ow2.weblab.service.exposer;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jws.WebService;

import org.apache.commons.logging.LogFactory;
import org.weblab_project.core.factory.AnnotationFactory;
import org.weblab_project.core.helper.PoKHelper;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.helper.ResourceHelper;
import org.weblab_project.core.model.Annotation;
import org.weblab_project.core.model.Document;
import org.weblab_project.core.ontologies.DublinCore;
import org.weblab_project.core.ontologies.WebLab;
import org.weblab_project.core.properties.PropertiesLoader;
import org.weblab_project.services.analyser.Analyser;
import org.weblab_project.services.analyser.ProcessException;
import org.weblab_project.services.analyser.types.ProcessArgs;
import org.weblab_project.services.analyser.types.ProcessReturn;
import org.weblab_project.services.exception.WebLabException;

/**
 * This service is used in conjunction with an Apache Tomcat or Apache server.
 * 
 * The purpose is to annotate documents in input with an "exposition URL" i.e. an URL that can be used in a web browser to access this document.
 * 
 * This service only works when processed file are exposed using the server and have the same naming pattern than the read annotation on the document (in most of the case, we use DC:source).
 * Some parameter can be set using a file name : local-file-exposer.config
 * 
 * <ul>
 * <li><b><tt>sourceUri</tt></b>: The predicate's URI of the statement that contains the original source of the Document to be modified. Default value is
 * <code>http://purl.org/dc/elements/1.1/source</code>.</li>
 * <li><b><tt>sourceIsResource</tt></b>: Sometimes the object of the statement containing the original source of the Document is a Resource (since an URL might be an URI); in this case this parameter
 * has to be true. Default value is <code>false</code>.</li>
 * <li><b><tt>exposedAsUri</tt></b>: The predicate's URI of the statement that will contain the exposed URL of the Document. Default value is
 * <code>http://weblab-project.org/core/model/property/isExposedAs</code>.</li>
 * <li><b><tt>nbStartCharToReplace</tt></b>: The number of character to be remove at the beginning of the String to replace; in most of the case it's the path to the whole folder that will be exposed.
 * Default value is <code>0</code>.</li>
 * <li><b><tt>byString</tt></b>: The String to be added at the beginning of the String to replace; in most of the case it's <tt>protocol://host:port/exposition_pattern/</tt>. Default value is
 * <code>""</code>.</li>
 * <li><b><tt>annotateAsUri</tt></b>: Whether the URL shall be annotated as Resource. If not it will be a literal statement. Default value is <code>false</code>.</li>
 * <li><b><tt>urlEncodeBetweenSlashes</tt></b>: Whether the URL shall be annotated as Resource. If not it will be a literal statement. Default value is <code>false</code>.</li>
 * <li><b><tt>urlEncoding</tt></b>: When encoding a URL it's needed to use the encoding used by the server for URL encoding; otherwise some file might never be exposed (especially files containing
 * accent in their name or other complex characters). Default value is <code>ISO-8859-1</code>.</li>
 * <li><b><tt>replacementForSpace</tt></b>: The java method for URL encoding replaces spaces by '+'. But servers like Tomcat or Apache are using "%20" (in UTF-8 and ISO-8859-1 for instance) as
 * replacement String. Default value is <code>%20</code>.</li>
 * </ul>
 * 
 * @author ymombrun
 * @date 24 oct. 08
 */
@WebService(endpointInterface = "org.weblab_project.services.analyser.Analyser")
public class LocalFileExposer implements Analyser {

	/**
	 * The predicate's URI of the statement that contains the original source of the Document to be modified.
	 */
	final protected String sourceUri;

	/**
	 * Sometimes the object of the statement containing the original source of the Document is a Resource (since an URL might be an URI); in this case this parameter has to be true.
	 */
	final protected boolean sourceIsResource;

	/**
	 * It's possible to change the URI of the predicate to be written with the exposition URL.
	 */
	final protected String exposedAsUri;

	/**
	 * The number of character to be remove at the beginning of the String to replace; in most of the case it's the path to the whole folder that will be exposed.
	 */
	final protected int nbStartCharToReplace;

	/**
	 * The String to be added at the beginning of the String to replace; in most of the case it's protocol://host:port/exposition_pattern/
	 */
	final protected String byString;

	/**
	 * Whether the URL shall be annotated as Resource. If not it will be a literal statement.
	 */
	final protected boolean annotateAsUri;

	/**
	 * Whether or not to URL encode between the slashes (and backslashes). It's needed in most of the case.
	 */
	final protected boolean urlEncodeBetweenSlashes;

	/**
	 * When encoding a URL it's needed to use the encoding used by the server for URL encoding; otherwise some file might never be exposed (especially files containing accent in their name or other
	 * complex characters).
	 */
	final protected String urlEncoding;

	/**
	 * The java method for URL encoding replaces spaces by '+'. But servers like Tomcat or Apache are using "%20" (in UTF-8 and ISO-8859-1 for instance) as replacement String.
	 */
	final protected String replacementForSpace;

	/**
	 * The name of the configuration file.
	 */
	final public static String CONFIG_FILE = "local-file-exposer.config";


	/**
	 * Constructors
	 */
	public LocalFileExposer() {
		final Map<String, String> props = PropertiesLoader.loadProperties(LocalFileExposer.CONFIG_FILE);

		if (props.containsKey("sourceUri")) {
			this.sourceUri = props.get("sourceUri");
		} else {
			LogFactory.getLog(this.getClass()).info("No sourceUri loaded from file: " + LocalFileExposer.CONFIG_FILE + ". " + DublinCore.SOURCE_PROPERTY_NAME + " used instead.");
			this.sourceUri = DublinCore.SOURCE_PROPERTY_NAME;
		}
		if (props.containsKey("sourceIsResource")) {
			this.sourceIsResource = Boolean.parseBoolean(props.get("sourceIsResource"));
		} else {
			LogFactory.getLog(this.getClass()).info("No sourceIsResource loaded from file: " + LocalFileExposer.CONFIG_FILE + ". false used instead.");
			this.sourceIsResource = false;
		}
		if (props.containsKey("exposedAsUri")) {
			this.exposedAsUri = props.get("exposedAsUri");
		} else {
			LogFactory.getLog(this.getClass()).info("No exposedAsUri loaded from file: " + LocalFileExposer.CONFIG_FILE + ". " + WebLab.IS_EXPOSED_AS + " used instead.");
			this.exposedAsUri = WebLab.IS_EXPOSED_AS;
		}
		if (props.containsKey("nbStartCharToReplace")) {
			int value = 0;
			try {
				value = Integer.parseInt(props.get("nbStartCharToReplace"));
			} catch (final NumberFormatException nfe) {
				LogFactory.getLog(this.getClass()).warn("nbStartCharToReplace not loaded from file: " + LocalFileExposer.CONFIG_FILE + ". 0 used instead.", nfe);
			}
			this.nbStartCharToReplace = value;
		} else {
			LogFactory.getLog(this.getClass()).info("No nbStartCharToReplace loaded from file: " + LocalFileExposer.CONFIG_FILE + ". 0 used instead.");
			this.nbStartCharToReplace = 0;
		}
		if (props.containsKey("byString")) {
			this.byString = props.get("byString");
		} else {
			LogFactory.getLog(this.getClass()).info("No byString loaded from file: " + LocalFileExposer.CONFIG_FILE + ". \"\" used instead.");
			this.byString = "";
		}
		if (props.containsKey("annotateAsUri")) {
			this.annotateAsUri = Boolean.parseBoolean(props.get("annotateAsUri"));
		} else {
			LogFactory.getLog(this.getClass()).info("No annotateAsUri loaded from file: " + LocalFileExposer.CONFIG_FILE + ". false used instead.");
			this.annotateAsUri = false;
		}
		if (props.containsKey("urlEncodeBetweenSlashes")) {
			this.urlEncodeBetweenSlashes = Boolean.parseBoolean(props.get("urlEncodeBetweenSlashes"));
		} else {
			LogFactory.getLog(this.getClass()).info("No urlEncodeBetweenSlashes loaded from file: " + LocalFileExposer.CONFIG_FILE + ". true used instead.");
			this.urlEncodeBetweenSlashes = true;
		}
		if (props.containsKey("urlEncoding")) {
			this.urlEncoding = props.get("urlEncoding");
		} else {
			LogFactory.getLog(this.getClass()).info("No urlEncoding loaded from file: " + LocalFileExposer.CONFIG_FILE + ". ISO-8859-1 used instead.");
			this.urlEncoding = "ISO-8859-1";
		}
		if (props.containsKey("replacementForSpace")) {
			this.replacementForSpace = props.get("replacementForSpace");
		} else {
			LogFactory.getLog(this.getClass()).info("No replacementForSpace loaded from file: " + LocalFileExposer.CONFIG_FILE + ". %20 used instead.");
			this.replacementForSpace = "%20";
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.weblab_project.services.analyser.Analyser#process(org.weblab_project.services.analyser.types.ProcessArgs)
	 */
	public ProcessReturn process(ProcessArgs args) throws ProcessException {
		Document doc = LocalFileExposer.checkArgs(args);

		ResourceHelper rh = RDFHelperFactory.getResourceHelper(doc);
		Set<String> sources;
		if (this.sourceIsResource) {
			sources = new HashSet<String>(rh.getRessOnPredSubj(doc.getUri(), this.sourceUri));
		} else {
			sources = new HashSet<String>(rh.getLitsOnPredSubj(doc.getUri(), this.sourceUri));
		}

		if (sources.isEmpty()) {
			LogFactory.getLog(this.getClass()).warn("No '" + this.sourceUri + "' property found in resource '" + doc.getUri() + "'. Nothing is done.");
			ProcessReturn pr = new ProcessReturn();
			pr.setResource(doc);
			return pr;
		}

		final String originalSource = sources.iterator().next();

		if (sources.size() > 1) {
			LogFactory.getLog(this.getClass()).warn("More than one value found for property '" + this.sourceUri + "' in resource '" + doc.getUri() + "'.");
			LogFactory.getLog(this.getClass()).debug("Values found were '" + sources + "'.");
			LogFactory.getLog(this.getClass()).debug("'" + originalSource + "' will be used.");
		}

		String tempSource;
		StringBuilder finalSource = new StringBuilder();
		if (this.nbStartCharToReplace > 1) {
			if (originalSource.length() > this.nbStartCharToReplace) {
				finalSource.append(this.byString);
				tempSource = originalSource.substring(this.nbStartCharToReplace);
			} else {


				LogFactory.getLog(this.getClass()).warn(
						"Source found (" + originalSource + ") is smaller than the number of character to replace (" + String.valueOf(this.nbStartCharToReplace) + "). Nothing is done.");
				ProcessReturn pr = new ProcessReturn();
				pr.setResource(doc);
				return pr;
			}
		} else {
			LogFactory.getLog(this.getClass()).debug("nbStartCharToReplace is smaller that one; no replacement is done.");
			tempSource = originalSource;
		}

		tempSource = tempSource.replace('\\', '/');

		if (this.urlEncodeBetweenSlashes) {
			try {
				final String[] splitted = tempSource.split("/");
				for (int i = 0; i < splitted.length; i++) {
					finalSource.append("/");
					finalSource.append(URLEncoder.encode(splitted[i], this.urlEncoding));
				}
			} catch (final UnsupportedEncodingException uee) {
				LogFactory.getLog(this.getClass()).fatal(uee);
				WebLabException wle = new WebLabException();
				wle.setErrorId("E0");
				wle.setErrorMessage("Unexpected error.");
				throw new ProcessException("Encoding '" + this.urlEncoding + "' is not supported.", wle, uee);

			}
		}

		final String exposedAs = finalSource.toString().replace("+", this.replacementForSpace);

		Annotation annot = AnnotationFactory.createAndLinkAnnotation(doc);
		PoKHelper pokH = RDFHelperFactory.getPoKHelper(annot);

		if (this.annotateAsUri) {
			pokH.createResStat(doc.getUri(), this.exposedAsUri, exposedAs);
		} else {
			pokH.createLitStat(doc.getUri(), this.exposedAsUri, exposedAs);
		}

		ProcessReturn pr = new ProcessReturn();
		pr.setResource(doc);
		return pr;
	}

	/**
	 * @param args
	 * @return
	 */
	private static Document checkArgs(ProcessArgs args) throws ProcessException {
		WebLabException wle = new WebLabException();
		wle.setErrorId("E1");
		wle.setErrorMessage("Invalid parameter.");
		if (args == null) {
			throw new ProcessException("ProcessArgs was null", wle);
		} else if (args.getResource() == null) {
			throw new ProcessException("Resource was null", wle);
		} else if (!(args.getResource() instanceof Document)) {
			throw new ProcessException("Resource was not a Document but a: " + args.getResource().getClass().getName(), wle);
		} else if (args.getResource().getAnnotation().isEmpty()) {
			throw new ProcessException("Document does not contains any Annotation.", wle);
		}
		return (Document) args.getResource();

	}

}
