/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service;

import java.io.File;
import java.util.Map;

import javax.jws.WebService;

import org.apache.commons.codec.digest.DigestUtils;
import org.weblab_project.core.exception.WebLabCheckedException;
import org.weblab_project.core.exception.WebLabUncheckedException;
import org.weblab_project.core.jaxb.WebLabMarshaler;
import org.weblab_project.core.model.Resource;
import org.weblab_project.core.properties.PropertiesLoader;
import org.weblab_project.services.exception.WebLabException;
import org.weblab_project.services.resourcecontainer.GetResourceException;
import org.weblab_project.services.resourcecontainer.ResourceContainer;
import org.weblab_project.services.resourcecontainer.SaveResourceException;
import org.weblab_project.services.resourcecontainer.types.GetResourceArgs;
import org.weblab_project.services.resourcecontainer.types.GetResourceReturn;
import org.weblab_project.services.resourcecontainer.types.SaveResourceArgs;
import org.weblab_project.services.resourcecontainer.types.SaveResourceReturn;

/**
 * This is a simple implementation of a ResourceContainer.
 * 
 * It enable to save resource on a file system and get them after.
 * It is a simpler version that the filerepository one.
 * Main differences are:
 * <ul>
 * <li>This version does not allocate uris; it means that incoming URIs has to be unique otherwise their will be some replacements. It maybe be useful for some applications but in most of the case
 * you'd rather let the FileRepository allocate URIs.</li>
 * <li>This version does not enable to get a subresource. It means that if you just want to change an annotation on a document, should shall re-save the whole document. The FileRepository has this
 * functionality.</li>
 * </ul>
 * 
 * @author ymombrun, Copyright EADS DS
 * @date 2010-04-15
 */
@WebService(endpointInterface = "org.weblab_project.services.resourcecontainer.ResourceContainer")
public class SimpleRepo implements ResourceContainer {

	/**
	 * The property file to be used to configure the service.
	 */
	public static final String PROPERTIES_FILE = "simple-repo.properties";

	/**
	 * Name of the property denoting the repository path.
	 */
	public final static String REPO_PATH_PROPERTY = "path";

	/**
	 * The base folder of the repository in the file system
	 */
	private final File repoBase;


	/**
	 * Instance of WebLabMarshaller used to read resources stored in repo.
	 */
	private final WebLabMarshaler reader;

	/**
	 * Instance of WebLabMarshaller used to store resources in repo.
	 */
	private final WebLabMarshaler writer;


	/**
	 * The default constructor
	 */
	public SimpleRepo() {
		Map<String, String> props = PropertiesLoader.loadProperties(PROPERTIES_FILE);

		final String path = props.get(REPO_PATH_PROPERTY);
		if (path == null) {
			throw new WebLabUncheckedException("Property " + REPO_PATH_PROPERTY + " not found or empty in file " + PROPERTIES_FILE + ".");
		}
		this.repoBase = new File(path);
		this.repoBase.mkdirs();
		this.reader = new WebLabMarshaler();
		this.writer = new WebLabMarshaler();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.weblab_project.services.resourcecontainer.ResourceContainer#getResource(org.weblab_project.services.resourcecontainer.types.GetResourceArgs)
	 */
	@Override
	public GetResourceReturn getResource(final GetResourceArgs args) throws GetResourceException {
		// Extract the resource to get
		final String uri = this.checkGetResourceArgs(args);

		// Get the abstract file path to read
		final File file = this.uriToFile(uri);

		// Extract the resource contained by file
		final Resource res = this.readFile(file, uri);

		GetResourceReturn grr = new GetResourceReturn();
		grr.setResource(res);
		return grr;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.weblab_project.services.resourcecontainer.ResourceContainer#saveResource(org.weblab_project.services.resourcecontainer.types.SaveResourceArgs)
	 */
	@Override
	public SaveResourceReturn saveResource(final SaveResourceArgs args) throws SaveResourceException {
		// Extract Resource to save
		final Resource res = this.checkSaveResourceArgs(args);

		// Get the abstract file path to write
		final File file = this.uriToFile(res.getUri());

		// Write the resource into the file
		this.writeFile(file, res);

		SaveResourceReturn srr = new SaveResourceReturn();
		srr.setResourceId(res.getUri());
		return srr;
	}


	/**
	 * @param file
	 *            The file that will contain resource
	 * @param res
	 *            The resource to store on file system
	 * @throws If
	 *             res was not marshalled into file (in most of the cast due to an IOException).
	 */
	private synchronized void writeFile(final File file, final Resource res) throws SaveResourceException {
		try {
			this.writer.marshalResource(res, file);
		} catch (final WebLabCheckedException wlce) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E0");
			wle.setErrorMessage("Unexpected error.");
			throw new SaveResourceException("Error when marshalling the resource with URI '" + res.getUri() + "'.", wle, wlce);
		}
	}

	/**
	 * @param args
	 *            The SaveResourceArgs to test
	 * @return The resource contained by args
	 * @throws SaveResourceException
	 *             If args, args.getResource() or args.getResource().getUri() is null.
	 */
	private Resource checkSaveResourceArgs(final SaveResourceArgs args) throws SaveResourceException {
		if (args == null) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new SaveResourceException("SaveResourceArgs was null", wle);
		}
		final Resource res = args.getResource();
		if (res == null) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new SaveResourceException("Resource in SaveResourceArgs was null", wle);
		}
		if (res.getUri() == null) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new SaveResourceException("URI of the Resource in SaveResourceArgs was null", wle);
		}
		return res;
	}

	/**
	 * @param file
	 *            The file to be unmarshalled into a resource
	 * @param uri
	 *            The uri to be used for logs
	 * @return The resource contained in file
	 * @throws GetResourceException
	 *             If was not unmarshalled (in most of the cast due to an IOException).
	 */
	private synchronized Resource readFile(final File file, final String uri) throws GetResourceException {
		if (!file.exists()) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new GetResourceException("No resource with '" + uri + "' as URI found.", wle);
		}
		try {
			return this.reader.unmarshal(file, Resource.class);
		} catch (final WebLabCheckedException wlce) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E0");
			wle.setErrorMessage("Unexpected error.");
			throw new GetResourceException("Error when unmarshalling the file with URI '" + uri + "'.", wle, wlce);
		}
	}

	/**
	 * @param uri
	 *            The URI of the Resource to get the storing file
	 * @return The file used to store a resource
	 */
	private File uriToFile(final String uri) {
		// Creates a String of 64 chars
		final String sha = DigestUtils.sha256Hex(uri);

		// To balance the files, in more than one directory, creates a directory having the first SHA char has name.
		final File folder = new File(this.repoBase, sha.substring(0, 1));
		folder.mkdirs();

		// The file in folder.
		final File file = new File(folder, sha + ".xml");

		return file;
	}

	/**
	 * @param args
	 *            Wrapper of resourceId
	 * @return The resourceId in args
	 * @throws GetResourceException
	 *             If args or resourceId in args is null.
	 */
	private String checkGetResourceArgs(final GetResourceArgs args) throws GetResourceException {
		if (args == null) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new GetResourceException("GetResourceArgs was null", wle);
		}
		final String uri = args.getResourceId();
		if (uri == null) {
			WebLabException wle = new WebLabException();
			wle.setErrorId("E1");
			wle.setErrorMessage("Invalid parameter.");
			throw new GetResourceException("ResourceId GetResourceArgs was null", wle);
		}
		return uri;
	}

}
