/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service.gazetteer.api;

import java.util.Date;

import org.apache.commons.logging.Log;
import org.weblab_project.core.factory.AnnotationFactory;
import org.weblab_project.core.factory.SegmentFactory;
import org.weblab_project.core.helper.PoKHelper;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.model.Annotation;
import org.weblab_project.core.model.Resource;
import org.weblab_project.core.model.text.LinearSegment;
import org.weblab_project.core.model.text.Text;
import org.weblab_project.core.ontologies.RDF;
import org.weblab_project.core.ontologies.RDFS;
import org.weblab_project.core.ontologies.WebLab;
import org.weblab_project.core.util.ResourceUtil;
import org.weblab_project.services.analyser.Analyser;
import org.weblab_project.services.analyser.ProcessException;
import org.weblab_project.services.analyser.types.ProcessArgs;
import org.weblab_project.services.analyser.types.ProcessReturn;

public abstract class TextAnnotator implements Analyser {

	protected Log logger;

	public TextAnnotator(Log logger) {
		super();
		this.logger = logger;
	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws ProcessException {
		Resource resource = args.getResource();

		for (Text text : ResourceUtil.getSelectedSubResources(resource, Text.class)) {
			this.logger.info("Processing text.");
			this.process(text);
		}

		ProcessReturn ret = new ProcessReturn();
		ret.setResource(resource);
		return ret;
	}

	public void annote(final Text text, final int offset, final String word, final String className, final String label) {
		this.logger.debug("Creating segment : " + offset + " - " + word);
		LinearSegment linearSegment = SegmentFactory.createAndLinkLinearSegment(text, offset, offset + word.length());
		Annotation annotation = AnnotationFactory.createAndLinkAnnotation(linearSegment);
		final String instanceUri = WebLab.PROCESSING_PROPERTY_NAMESPACE + "temp#" + new Date().getTime();

		PoKHelper h = RDFHelperFactory.getPoKHelper(annotation);
		h.setAutoCommitMode(false);
		h.createResStat(linearSegment.getUri(), WebLab.REFERS_TO, instanceUri);
		h.createResStat(instanceUri, RDF.TYPE, WebLab.PROCESSING_PROPERTY_NAMESPACE + "demo#" + className);
		h.createLitStat(instanceUri, RDFS.LABEL, label + " -> " + className);
		h.createLitStat(instanceUri, WebLab.IS_CANDIDATE, "true");

		h.commit();

		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Word annotated : " + word + " class " + className + " -> " + offset);
			this.logger.debug("In text : '" + text.getContent().substring(offset, offset + word.length()) + "'");
		}

	}

	public abstract void process(Text text) throws ProcessException;

}
