/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.searcher.impl;

import java.io.File;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.jws.WebService;
import javax.servlet.ServletContext;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.util.search.SearcherConfig;
import org.weblab_project.core.exception.WebLabCheckedException;
import org.weblab_project.core.factory.ResourceFactory;
import org.weblab_project.core.helper.BeanHelper;
import org.weblab_project.core.model.PieceOfKnowledge;
import org.weblab_project.core.model.query.Query;
import org.weblab_project.core.model.query.ResultSet;
import org.weblab_project.core.model.query.StringQuery;
import org.weblab_project.core.util.PoKUtil;
import org.weblab_project.core.util.ResourceUtil;
import org.weblab_project.services.searcher.SearchException;
import org.weblab_project.services.searcher.Searcher;
import org.weblab_project.services.searcher.types.SearchArgs;
import org.weblab_project.services.searcher.types.SearchReturn;

/**
 * FastSolrSearcher is a searcher with no Jena helper. RDF is generated manually. Meta enrichment is not possible with
 * this version of search
 */

@WebService(endpointInterface = "org.weblab_project.services.searcher.Searcher")
public class FastSolrSearcher implements Searcher {

	@javax.annotation.Resource
	protected WebServiceContext wsContext;

	private static int resultsCpt = 0;

	private SolrSearcherConfig conf;

	private SolrComponent solrComponent;

	public FastSolrSearcher() {
		if (this.conf == null) {
			this.conf = BeanHelper.getInstance().getSpecificInstance(SolrSearcherConfig.DEFAULT_CONF_FILE, true).getBean(
					SearcherConfig.DEFAULT_BEAN_NAME, SolrSearcherConfig.class);
		}
		if (this.solrComponent == null)
			this.solrComponent = SolrComponent.getInstance();
	}

	@Override
	public SearchReturn search(SearchArgs arg0) throws SearchException {
		Query q = arg0.getQuery();
		SearchReturn re = new SearchReturn();
		re.setResultSet(this.search(q, arg0.getOffset(), arg0.getLimit()));
		return re;
	}

	/**
	 * Use <code>SolrComponent</code> for Solr index querying and format response in RDF manually.<br/>
	 * The ResultSet contains a Pok which himself contains Hits.<br/>
	 * 
	 * @param q
	 *            the Weblab <code>StringQuery</code>
	 * @param offset
	 *            results start index
	 * @param limit
	 *            results end index
	 * @return the <code>ResultSet</code> containing hits
	 * @throws SearchException
	 */
	public ResultSet search(Query q, int offset, int limit) throws SearchException {
		StringQuery query = null;

		if (q instanceof StringQuery) {
			query = (StringQuery) q;
		}
		else {
			throw new SearchException("This service " + this.getClass().getSimpleName() + " can only process "
					+ StringQuery.class.getSimpleName() + " and not " + q.getClass().getSimpleName(), null);
		}

		try {
			LogFactory.getLog(this.getClass()).debug("query request : " + query.getRequest());
			LogFactory.getLog(this.getClass()).debug("Query input : " + ResourceUtil.saveToXMLString(query));
		}
		catch (WebLabCheckedException e) {
			LogFactory.getLog(this.getClass()).debug("Error when logging query input", e);
		}

		int resultId = resultsCpt++;
		ResultSet results = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF,
				SolrSearcherConfig.DEFAULT_IDRES + resultId, ResultSet.class);
		PieceOfKnowledge pok = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF,
				SearcherConfig.DEFAULT_IDPOK + resultId, PieceOfKnowledge.class);
		results.setPok(pok);

		try {
			/*
			 * Open Solr searcher
			 */
			this.solrComponent.open(this.getWebAppPath(), "core0");

			QueryResponse res = this.solrComponent.search(query.getRequest(), offset, limit);

			int resCpt = 0;

			SolrDocumentList resultDocs = res.getResults();
			Map<String, Map<String, List<String>>> highlightings = res.getHighlighting();

			StringBuffer buf = new StringBuffer();

			buf.append("<rdf:RDF ");
			buf.append("		xmlns:rdf=\"http://www.w3.org/1999/02/22-rdf-syntax-ns#\" ");
			buf.append("		xmlns:rdfs=\"http://www.w3.org/2000/01/rdf-schema#\" ");
			buf.append("		xmlns:dc=\"http://purl.org/dc/elements/1.1/\" ");
			buf.append("		xmlns:dct=\"http://purl.org/dc/terms/\" ");
			buf.append("		xmlns:wl=\"http://weblab-project.org/core/model/property/\" ");
			buf.append("		xmlns:wls=\"http://weblab-project.org/core/model/property/retrieval/\">");

			// ResultSet description
			buf.append("<rdf:Description rdf:about=\"weblab://eads.indexsearch.lucene/result" + resultId + "\">");
			buf.append("	<rdf:type rdf:resource=\"http://weblab-project.org/core/model/class/ResultSet\"/>");
			for (Entry<String, String> local : this.conf.getResultSetLabels().entrySet()) {
				buf.append("	<rdfs:label xml:lang=\""+local.getKey()+"\">"+local.getValue()+"</rdfs:label>");
			}
			buf.append("    <wl:offset> " + offset + " </wl:offset>");
			buf.append("	<wl:hasNumberOfResults> " + resultDocs.getNumFound() + "</wl:hasNumberOfResults>");
			buf.append("	<wl:isResultOf rdf:resource=\"" + q.getUri() + "\" />");
			buf.append("</rdf:Description>");

			// create a Hit for each result
			for (SolrDocument hit : resultDocs) {
				String resourceUri = String.valueOf(hit.getFieldValue("id"));
				String hitUri = "weblab://" + SolrSearcherConfig.DEFAULT_IDREF + "/" + SearcherConfig.DEFAULT_IDHIT
						+ resCpt;

				buf.append("<rdf:Description rdf:about=\"" + hitUri + "\" >");
				buf.append("<rdf:type rdf:resource=\"" + this.conf.getHitClass() + "\" />");
				buf.append("<wls:hasRank>" + (offset + resCpt + 1) + "</wls:hasRank>");
				buf.append("<wls:hasScore>" + hit.getFieldValue("score") + "</wls:hasScore>");
				if (highlightings != null) {
					if (highlightings.get(hit.getFieldValue("id")).size() > 0) {
						List<String> hitHighlightingList = highlightings.get(hit.getFieldValue("id")).get("text");
						StringBuffer snippet = new StringBuffer();
						for (String hl : hitHighlightingList) {
							snippet.append(hl);
						}
						buf.append("<wls:hasDescription>" + snippet.toString() + "</wls:hasDescription>");
					}
				}
				buf.append("<wls:isLinkedTo rdf:resource=\"" + resourceUri + "\" />");
				buf.append("<wls:inResultSet rdf:resource=\"" + results.getUri() + "\" />");
				buf.append("</rdf:Description>");

				resCpt++;
			}

			buf.append("</rdf:RDF>");

			PoKUtil.setPoKData(pok, buf.toString());
		}
		catch (WebLabCheckedException e) {
			throw new SearchException("Cannot retrieve the results to query [" + query.getRequest() + "]", null);
		}

		return results;
	}

	/**
	 * Return the Web application path
	 * 
	 * @return webapp path
	 * @throws WebLabCheckedException
	 */
	private String getWebAppPath() throws WebLabCheckedException {
		String appPath = "./src/main/resources/";
		if (this.wsContext != null) {
			ServletContext ctx = (ServletContext) this.wsContext.getMessageContext().get(MessageContext.SERVLET_CONTEXT);
			appPath = ctx.getRealPath("WEB-INF/");
		}
		else {
			LogFactory.getLog(FastSolrSearcher.class).warn(
					"Webservice context not available returning current local path as default web app path");
		}

		File f = new File(appPath);
		if (!f.exists())
			throw new WebLabCheckedException("Webapp path [" + appPath + "] does not exists...");
		return appPath;
	}
}
