/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.indexer.impl;

import java.io.File;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;
import javax.servlet.ServletContext;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.services.solr.SolrComponent;
import org.weblab_project.core.exception.WebLabCheckedException;
import org.weblab_project.core.model.MediaUnit;
import org.weblab_project.core.model.multimedia.BinaryMediaUnit;
import org.weblab_project.services.exception.WebLabException;
import org.weblab_project.services.indexer.IndexException;
import org.weblab_project.services.indexer.Indexer;
import org.weblab_project.services.indexer.types.IndexArgs;
import org.weblab_project.services.indexer.types.IndexReturn;

/**
 * Indexer using Embedded Solr server. This class only check index arguments and call Solr component
 */

@WebService(endpointInterface = "org.weblab_project.services.indexer.Indexer")
public class SolrIndexer implements Indexer {

	@javax.annotation.Resource
	protected WebServiceContext wsContext;

	private SolrComponent solr;

	@PostConstruct
	public void init() {
		this.solr = SolrComponent.getInstance();
	}

	@Override
	public IndexReturn index(IndexArgs args) throws IndexException {
		final MediaUnit unit = SolrIndexer.checkArgs(args);
		if (unit instanceof BinaryMediaUnit) {
			LogFactory.getLog(this.getClass()).warn(
					"MediaUnit of type: " + unit.getClass().getSimpleName() + " cannot be indexed with Solr.");
			return new IndexReturn();
		}
		try {
			// This condition is for JUnit tests which not call PostConstruct method
			if (this.solr == null)
				this.solr = SolrComponent.getInstance();
			
			this.solr.open(this.getWebAppPath());
			this.solr.addDocument(unit);
		}
		catch (WebLabCheckedException e) {
			WebLabException excp = new WebLabException();
			excp.setErrorMessage("Cannot open Solr index : " + e.getMessage());
			throw new IndexException("Cannot open Solr index.", excp);
		}
		return new IndexReturn();
	}

	@PreDestroy
	public void destroy() {
		this.solr.close();
	}

	/**
	 * Return the Web application path
	 * 
	 * @return webapp path
	 * @throws WebLabCheckedException
	 */
	private String getWebAppPath() throws WebLabCheckedException {
		String appPath = "./src/main/resources";
		if (this.wsContext != null) {
			ServletContext ctx = (ServletContext) this.wsContext.getMessageContext().get(MessageContext.SERVLET_CONTEXT);
			appPath = ctx.getRealPath("WEB-INF/classes");
		}
		else {
			LogFactory.getLog(SolrIndexer.class).warn(
					"Webservice context not available returning current local path as default web app path");
		}

		File f = new File(appPath);
		if (!f.exists())
			throw new WebLabCheckedException("Webapp path [" + appPath + "] does not exists...");
		return appPath;
	}

	/**
	 * @param args
	 *            The <code>IndexArgs</code> to check in the begin of <code>index</code>.
	 * @return The contained <code>MediaUnit</code>
	 * @throws IndexException
	 *             if we are unable to extract the contained <code>MediaUnit</code>
	 */
	private static MediaUnit checkArgs(final IndexArgs args) throws IndexException {
		WebLabException wle = new WebLabException();
		wle.setErrorId("E1");
		wle.setErrorMessage("Invalid parameter");
		if (args == null) {
			LogFactory.getLog(SolrIndexer.class).error("IndexArgs was null.");
			throw new IndexException("IndexArgs was null.", wle);
		}
		if (args.getResource() == null) {
			LogFactory.getLog(SolrIndexer.class).error("Args must contain a non-null Resource to index");
			throw new IndexException("Args must contain a " + "non-null Resource to index", wle);
		}
		if (!(args.getResource() instanceof MediaUnit)) {
			LogFactory.getLog(SolrIndexer.class).error("Resource to index as not a MediaUnit.");
			throw new IndexException("Resource to index " + "as not a MediaUnit.", wle);
		}
		return (MediaUnit) args.getResource();
	}

	/**
	 * @return the wsContext
	 */
	public WebServiceContext getWsContext() {
		return this.wsContext;
	}

	/**
	 * @param wsContext the wsContext to set
	 */
	public void setWsContext(WebServiceContext wsContext) {
		this.wsContext = wsContext;
	}
}
