/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.searcher.impl;

import java.io.File;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.jws.WebService;
import javax.servlet.ServletContext;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.util.index.Field;
import org.ow2.weblab.util.search.SearcherConfig;
import org.weblab_project.core.exception.WebLabCheckedException;
import org.weblab_project.core.factory.ResourceFactory;
import org.weblab_project.core.helper.BeanHelper;
import org.weblab_project.core.helper.PoKHelperExtended;
import org.weblab_project.core.helper.RDFHelperFactory;
import org.weblab_project.core.model.PieceOfKnowledge;
import org.weblab_project.core.model.query.Query;
import org.weblab_project.core.model.query.ResultSet;
import org.weblab_project.core.model.query.StringQuery;
import org.weblab_project.core.ontologies.RDF;
import org.weblab_project.core.ontologies.RDFS;
import org.weblab_project.core.ontologies.WebLab;
import org.weblab_project.core.util.ResourceUtil;
import org.weblab_project.services.searcher.SearchException;
import org.weblab_project.services.searcher.Searcher;
import org.weblab_project.services.searcher.types.SearchArgs;
import org.weblab_project.services.searcher.types.SearchReturn;

/**
 * Searcher using Embedded Solr server.<br/>
 * RDF output is generated with Jena and can be enriched with metadata stored in index. This should be define in
 * searcher configuration file.
 */

@WebService(endpointInterface = "org.weblab_project.services.searcher.Searcher")
public class SolrSearcher implements Searcher {

	@javax.annotation.Resource
	protected WebServiceContext wsContext;

	private static int resultsCpt = 0;

	private SolrSearcherConfig conf;

	private SolrComponent solrComponent;

	public SolrSearcher() {
		this.conf = BeanHelper.getInstance().getSpecificInstance(SolrSearcherConfig.DEFAULT_CONF_FILE, true).getBean(
				SearcherConfig.DEFAULT_BEAN_NAME, SolrSearcherConfig.class);
		this.solrComponent = SolrComponent.getInstance();
	}

	@Override
	public SearchReturn search(SearchArgs arg0) throws SearchException {
		Query q = arg0.getQuery();
		
		Integer offset;
		if (arg0.getOffset() != null && arg0.getOffset() > 0)
			offset = arg0.getOffset();
		else
			offset = new Integer(0);
		
		Integer limit;
		if ((arg0.getLimit() != null && arg0.getLimit() > 0))
			limit = arg0.getLimit();
		else
			limit = Integer.MAX_VALUE;
		
		SearchReturn re = new SearchReturn();
		re.setResultSet(this.search(q, offset, limit));
		return re;
	}

	/**
	 * Use <code>SolrComponent</code> for Solr index querying and format response in RDF with Jena.<br/>
	 * The ResultSet contains a Pok which himself contains Hits.<br/>
	 * If enrichment is activated, all fields presents in index and search configuration file are annotated.
	 * 
	 * @param q
	 *            the Weblab <code>StringQuery</code>
	 * @param offset
	 *            results start index
	 * @param limit
	 *            results end index
	 * @return the <code>ResultSet</code> containing hits
	 * @throws SearchException
	 */
	public ResultSet search(Query q, int offset, int limit) throws SearchException {
		StringQuery query = null;

		if (q instanceof StringQuery) {
			query = (StringQuery) q;
		}
		else {
			throw new SearchException("This service " + this.getClass().getSimpleName() + " can only process "
					+ StringQuery.class.getSimpleName() + " and not " + q.getClass().getSimpleName(), null);
		}

		try {
			LogFactory.getLog(this.getClass()).debug("query request : " + query.getRequest());
			LogFactory.getLog(this.getClass()).debug("Query input : " + ResourceUtil.saveToXMLString(query));
		}
		catch (WebLabCheckedException e) {
			LogFactory.getLog(this.getClass()).debug("Error when logging query input", e);
		}

		SearchReturn returnedResults = new SearchReturn();
		ResultSet results = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF,
				SolrSearcherConfig.DEFAULT_IDRES + resultsCpt++, ResultSet.class);
		PieceOfKnowledge pok = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF,
				SearcherConfig.DEFAULT_IDPOK + resultsCpt++, PieceOfKnowledge.class);
		results.setPok(pok);

		PoKHelperExtended h = RDFHelperFactory.getPoKHelperExtended(pok);
		h.setAutoCommitMode(false);
		returnedResults.setResultSet(results);

		/*
		 * Set namespace prefixes
		 */
		for (Iterator<String> nsPrefixIterator = this.conf.getNsPrefixProperties().keySet().iterator(); nsPrefixIterator
				.hasNext();) {
			String nsPrefix = nsPrefixIterator.next();
			h.setNSPrefix(nsPrefix, this.conf.getNsPrefixProperties().get(nsPrefix));
		}

		/*
		 * Set Result type property
		 */
		h.createResStat(results.getUri(), RDF.TYPE, WebLab.getUriFromClass(ResultSet.class));

		/*
		 * Set ResultSet "isResultOf" property
		 */
		h.createResStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.RESULTS_FROM_KEY), q.getUri());
		
		/*
		 *  Set ResultSet "rdf:label" property
		 */
		for (Entry<String, String> local : this.conf.getResultSetLabels().entrySet()) {
			h.createLitStat(results.getUri(), RDFS.LABEL, local.getValue(), local.getKey());
		}
		
		try {
			/*
			 * Open Solr searcher
			 */
			this.solrComponent.open(this.getWebAppPath());

			QueryResponse res = this.solrComponent.search(query.getRequest(), offset, limit);
			
			int resCpt = 0;

			// create a Document for each result
			SolrDocumentList resultDocs = res.getResults();
			Map<String, Map<String, List<String>>> highlightings = res.getHighlighting();

			if (resultDocs != null) {
				for (SolrDocument hit : resultDocs) {
					String resourceUri = String.valueOf(hit.getFieldValue("id"));
					String hitUri = "weblab://" + SolrSearcherConfig.DEFAULT_IDREF + "/" + SearcherConfig.DEFAULT_IDHIT
							+ resCpt;
	
					h.createResStat(hitUri, RDF.TYPE, this.conf.getHitClass());
	
					// adding hasRank annotation
					h.createLitStat(hitUri, this.conf.getProperties().get(SearcherConfig.HAS_RANK_KEY), String
							.valueOf(offset + resCpt + 1));
	
					// Solr score
					h.createLitStat(hitUri, this.conf.getProperties().get("hasScore"), String
							.valueOf(hit.getFieldValue("score")));
	
					// Generate snippet from hit highlightings
					if (highlightings != null) {
						if (highlightings.get(hit.getFieldValue("id")).size() > 0) {
							List<String> hitHighlightingList = highlightings.get(hit.getFieldValue("id")).get("text");
							StringBuffer snippet = new StringBuffer();
							for (String hl : hitHighlightingList) {
								snippet.append(hl);
							}
							h.createLitStat(hitUri, this.conf.getProperties().get(SearcherConfig.DESCRIPTION_KEY), snippet
									.toString());
						}
					}
	
					h.createResStat(hitUri, this.conf.getProperties().get(SearcherConfig.LINKED_TO_KEY), resourceUri);
	
					h.createResStat(hitUri, this.conf.getProperties().get(SearcherConfig.IN_RESULT_SET_KEY), results.getUri());
	
					// Meta enrichment
					if (this.conf.isHitsEnrichedWithMetas()) {
						Map<String, Collection<Object>> fields = hit.getFieldValuesMap();
						
						for (Entry<String, Field> confField : this.conf.getFieldsToRetrieve().entrySet()) {
							Field field = confField.getValue();
	
							if (fields.containsKey(field.getName())) {
								for (String property : field.getProperties()) {
									Collection<Object> fieldValues = fields.get(field.getName());
									for (Object fieldValue : fieldValues) {
										if (fieldValue instanceof String) {
											if (field.getType().equals("URI"))
												h.createResStat(resourceUri, property, ((String) fieldValue).trim());
											else
												h.createLitStat(resourceUri, property, ((String) fieldValue).trim());
										}
										else if (fieldValue instanceof Date)
											h.createLitStat(resourceUri, property, DateFormatUtils.ISO_DATETIME_TIME_ZONE_FORMAT
													.format((Date) fieldValue));
									}
								}
							}
						}
					}
	
					resCpt++;
				}
				h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.NB_RESULTS_KEY), String
						.valueOf(resultDocs.getNumFound()));
			}
			else {
				h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.NB_RESULTS_KEY), "0");
			}

			h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.OFFSET_KEY), String
					.valueOf(offset));

			h.commit();
		}
		catch (WebLabCheckedException e) {
			throw new SearchException("Cannot retrieve the results to query [" + query.getRequest() + "] - " + e.getMessage(), null);
		}

		return results;
	}

	/**
	 * Return the Web application path
	 * 
	 * @return webapp path
	 * @throws WebLabCheckedException
	 */
	private String getWebAppPath() throws WebLabCheckedException {
		String appPath = "./src/main/resources";
		if (this.wsContext != null) {
			ServletContext ctx = (ServletContext) this.wsContext.getMessageContext().get(MessageContext.SERVLET_CONTEXT);
			appPath = ctx.getRealPath("WEB-INF/classes");
		}
		else {
			LogFactory.getLog(SolrSearcher.class).warn(
					"Webservice context not available returning current local path as default web app path");
		}

		File f = new File(appPath);
		if (!f.exists())
			throw new WebLabCheckedException("Webapp path [" + appPath + "] does not exists...");
		return appPath;
	}

	/**
	 * @return the wsContext
	 */
	public WebServiceContext getWsContext() {
		return this.wsContext;
	}

	/**
	 * @param wsContext the wsContext to set
	 */
	public void setWsContext(WebServiceContext wsContext) {
		this.wsContext = wsContext;
	}

}
