/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.indexer.impl;

import java.io.File;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;
import javax.servlet.ServletContext;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.services.Indexer;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.indexer.IndexArgs;
import org.ow2.weblab.core.services.indexer.IndexReturn;
import org.ow2.weblab.services.solr.SolrComponent;

/**
 * Indexer using Embedded Solr server. This class only check index arguments and
 * call Solr component
 */

@WebService(endpointInterface = "org.ow2.weblab.core.services.Indexer")
public class SolrIndexer implements Indexer {

	@javax.annotation.Resource
	protected WebServiceContext wsContext;
	private SolrComponent solr;

	/**
	 * @param args
	 *            The <code>IndexArgs</code> to check in the begin of
	 *            <code>index</code>.
	 * @return The contained <code>MediaUnit</code>
	 * @throws IndexException
	 *             if we are unable to extract the contained
	 *             <code>MediaUnit</code>
	 */
	private static MediaUnit checkArgs(final IndexArgs args)
			throws InvalidParameterException {
		/*
		 * WebLabException wle = new WebLabException(); wle.setErrorId("E1");
		 * wle.setErrorMessage("Invalid parameter");
		 */
		if (args == null) {
			LogFactory.getLog(SolrIndexer.class).error("IndexArgs was null.");
			throw new InvalidParameterException("IndexArgs was null.",
					"IndexArgs was null.");
		}
		if (args.getResource() == null) {
			LogFactory.getLog(SolrIndexer.class).error(
					"Args must contain a non-null Resource to index");
			throw new InvalidParameterException("Args must contain a "
					+ "non-null Resource to index", "Args must contain a "
					+ "non-null Resource to index");
		}
		if (!(args.getResource() instanceof MediaUnit)) {
			LogFactory.getLog(SolrIndexer.class).error(
					"Resource to index as not a MediaUnit.");
			throw new InvalidParameterException("Resource to index "
					+ "as not a MediaUnit.", "Resource to index "
					+ "as not a MediaUnit.");
		}
		return (MediaUnit) args.getResource();
	}

	@PostConstruct
	public void init() {
		this.solr = SolrComponent.getInstance();
	}

	@Override
	public IndexReturn index(IndexArgs args) throws UnexpectedException,
			InvalidParameterException {
		final MediaUnit unit = SolrIndexer.checkArgs(args);
		// if (unit instanceof BinaryMediaUnit) {
		// LogFactory.getLog(this.getClass()).warn(
		// "MediaUnit of type: " + unit.getClass().getSimpleName() +
		// " cannot be indexed with Solr.");
		// return new IndexReturn();
		// }
		try {
			// This condition is for JUnit tests which not call PostConstruct
			// method
			if (this.solr == null)
				this.solr = SolrComponent.getInstance();

			this.solr.open(this.getWebAppPath());
			this.solr.addDocument(unit);
		} catch (WebLabCheckedException e) {

			throw new UnexpectedException("Cannot open Solr index.",
					e.getMessage());
		}
		return new IndexReturn();
	}

	@PreDestroy
	public void destroy() {
		if (this.solr != null)
			this.solr.close();
	}

	/**
	 * Return the Web application path
	 * 
	 * @return webapp path
	 * @throws WebLabCheckedException
	 */
	protected String getWebAppPath() throws WebLabCheckedException {
		String appPath = null;
		if (this.wsContext != null) {
			ServletContext ctx = (ServletContext) this.wsContext
					.getMessageContext().get(MessageContext.SERVLET_CONTEXT);
			appPath = ctx.getRealPath("WEB-INF/classes");
		} else {
			LogFactory
					.getLog(SolrIndexer.class)
					.warn("Webservice context not available returning current local path as default web app path");
		}
		
		if (appPath == null){
			throw new WebLabCheckedException("Webapp path [" + appPath
					+ "] is null...");
		}else{
			if(!new File(appPath).exists())
				throw new WebLabCheckedException("Webapp path [" + appPath
						+ "] does not exists...");
		}

		return appPath;
	}

	/**
	 * @return the wsContext
	 */
	public WebServiceContext getWsContext() {
		return this.wsContext;
	}

	/**
	 * @param wsContext
	 *            the wsContext to set
	 */
	public void setWsContext(WebServiceContext wsContext) {
		this.wsContext = wsContext;
	}
}
