/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.repository.file;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Properties;

import javax.xml.bind.JAXB;

import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Resource;

/**
 * Simple implementation of a file repository Save each resource to a File.
 * 
 * @author Jérémie Doucy, EADS DS IPCC
 */
public class FileRepository {

	protected File pathToFS;

	protected final static String DEFAULT_REPO_ID = "fileRepository";
	protected String repositoryID = DEFAULT_REPO_ID;

	protected final static String DEFAULT_FILE_PREFIX = "res_";
	protected String filePrefix = DEFAULT_FILE_PREFIX;

	protected final static int DEFAULT_FILE_PER_FOLDER = 100;
	protected int filePerFolder = DEFAULT_FILE_PER_FOLDER;

	protected File currentFolder;
	protected final WebLabMarshaller wlm = new WebLabMarshaller();

	public Resource getResource(final String uri) throws WebLabCheckedException {
		final RepoRI wlRi = new RepoRI(uri);
		Resource res;
		try {
			res = this.loadFile(wlRi.getIdResource() + ".xml");
		} catch (final FileNotFoundException fnfe) {
			throw new WebLabCheckedException("Uri '" + uri
					+ "' is not present in the repository.", fnfe);
		}
		final Resource result = this.getResourceInside(res, uri);
		if (result == null) {
			throw new WebLabCheckedException("Uri '" + uri
					+ "' is not present in the repository.");
		}
		return result;
	}

	public String saveResource(Resource resource) throws WebLabCheckedException {
		final Resource res = this.changeUri(resource);

		
		// writing the file
		try {
			this.writeFile(new RepoRI(res.getUri()), res);
		} catch (final FileNotFoundException fnfe) {
			throw new WebLabCheckedException("Unable to write resource: "
					+ resource.getUri() + " into resource: " + res.getUri()
					+ ".", fnfe);
		}
		return res.getUri();
	}

	/**
	 * recursive route which return resource corresponding to the uri
	 * 
	 * @param res
	 *            main resource
	 * @param uri
	 *            uri to find
	 * @return res corresponding to the uri
	 */
	protected Resource getResourceInside(final Resource res, final String uri)
			throws WebLabCheckedException {
		return this.getResourceInside(res, uri, null);
	}

	/**
	 * recursive route which replace newRes in res
	 * 
	 * @param res
	 *            main resource
	 * @param uri
	 *            uri to find (the same as in newRes)
	 * @param newRes
	 *            new resource which going to replace the funded one
	 * @return
	 */
	protected Resource getResourceInside(final Resource res, final String uri,
			final Resource newRes) throws WebLabCheckedException {
		return this.getResourceInside(res, uri, newRes, null);
	}

	protected Resource getResourceInside(final Resource res, final String uri,
			final Resource newRes, final Resource father)
			throws WebLabCheckedException {

		/*
		 * if it's the current return it
		 */
		if (res.getUri().equals(uri)) {
			if (newRes != null) {
				this.replaceResource(father, res, newRes);
			}
			return res;
		}

		/*
		 * recursive call for annotations tests
		 */
		List<Annotation> listAnnotCopy = new ArrayList<Annotation>(res
				.getAnnotation());
		Resource ret = null;
		Iterator<Annotation> itAnnot = listAnnotCopy.iterator();
		while (ret == null && itAnnot.hasNext()) {
			ret = this.getResourceInside(itAnnot.next(), uri, newRes, res);
		}

		if (ret != null) {
			return ret;
		}

		/*
		 * recursive call for mediaUnits tests
		 */
		if (res instanceof Document) {
			Document document = (Document) res;
			List<MediaUnit> listMediaUnitCopy = new ArrayList<MediaUnit>(
					document.getMediaUnit());
			Iterator<MediaUnit> itMed = listMediaUnitCopy.iterator();
			while (ret == null && itMed.hasNext()) {
				ret = this.getResourceInside(itMed.next(), uri, newRes, res);
			}

			if (ret != null) {
				return ret;
			}
		}

		return ret;
	}

	protected void replaceResource(final Resource father,
			final Resource oldRes, final Resource newRes)
			throws WebLabCheckedException {
		/*
		 * try to replace the founded one in the list
		 */

		if (newRes instanceof MediaUnit && father instanceof Document) {
			final MediaUnit newMed = (MediaUnit) newRes;
			final Document fatherDoc = (Document) father;
			final int index = fatherDoc.getMediaUnit().indexOf(oldRes);
			fatherDoc.getMediaUnit().remove(oldRes);
			fatherDoc.getMediaUnit().add(index, newMed);
		} else if (newRes instanceof Annotation) {
			final Annotation newAnnot = (Annotation) newRes;
			final int index = father.getAnnotation().indexOf(oldRes);
			father.getAnnotation().remove(oldRes);
			father.getAnnotation().add(index, newAnnot);
		} else {
			throw new WebLabCheckedException(
					"In the document repository, the uri '" + newRes.getUri()
							+ "' is a '" + oldRes.getClass()
							+ "' and is not replacable by a '"
							+ newRes.getClass() + "'.");
		}
	}

	// TODO Remove synchronized when WebLabMarshaller will be Thread safe.

	/**
	 * load the resource corresponding to the file name
	 * 
	 * @param fileName
	 * @return the loaded resource
	 * @throws WebLabCheckedException
	 * @throws FileNotFoundException
	 */
	protected synchronized Resource loadFile(final String fileName)
			throws WebLabCheckedException, FileNotFoundException {
		final FileInputStream fis = new FileInputStream(this.pathToFS
				.getAbsoluteFile()
				+ "/" + fileName);
		final Resource res;
		try {
			res = this.wlm.unmarshal(fis, Resource.class);
		} finally {
			try {
				fis.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(this.getClass()).warn(
						"Unable to close stream.");
			}
		}
		return res;
	}

	/**
	 * Set the path to the folder file System Creates it if not exist
	 * 
	 * @param pathToFS
	 *            path to the folder file system
	 * @throws WebLabCheckedException
	 */
	public void setPathToFS(final File pathToFS) throws WebLabCheckedException {
		if (pathToFS == null) {
			throw new WebLabCheckedException("Folder is null.");
		}

		/*
		 * check if pathToFS exist
		 */
		if (pathToFS.exists()) {
			/*
			 * check if it's a directory
			 */
			if (pathToFS.isDirectory()) {
				this.pathToFS = pathToFS;
			} else {
				throw new WebLabCheckedException("'"
						+ pathToFS.getAbsolutePath() + "' is not a folder.");
			}
		} else {
			/*
			 * try to create the folder
			 */
			try {
				if (pathToFS.mkdirs()) {
					this.pathToFS = pathToFS;
				} else {
					throw new WebLabCheckedException(
							"Unable to create folder : '"
									+ pathToFS.getAbsolutePath() + "'");
				}
			} catch (final SecurityException se) {
				throw new WebLabCheckedException("Unable to create folder: '"
						+ pathToFS.getAbsolutePath() + "'.", se);
			}
		}
		this.currentFolder = this.createNewFolder();

	}

	/**
	 * check if the resource is already present if not create a unique wlri and
	 * change it in the resource
	 * 
	 * @param res
	 *            resource
	 * @return new unique wlri
	 * @throws WebLabCheckedException
	 */
	protected Resource changeUri(final Resource res)
			throws WebLabCheckedException {
		Resource ret;
		/*
		 * the file name is <id-resource>.xml
		 */
		RepoRI wlRi;
		try {
			wlRi = new RepoRI(res.getUri());
		} catch (final WebLabCheckedException wlce) {
			LogFactory.getLog(this.getClass()).warn(
					"Unable to load a valid WLRI: '" + res.getUri() + "'.",
					wlce);
			wlRi = null;
		}

		/*
		 * first, is the isRef the current one
		 */
		if (wlRi != null && wlRi.getIdReference().equals(this.repositoryID)) {
			/*
			 * check if resource is already present
			 */
			// boolean present = false;
			// for (File file : currentFolder.listFiles()) {
			// String name = file.getName();
			// /*
			// * resource is present
			// */
			// if ((currentFolder.getName() + "/" + name.substring(0, name
			// .length() - 4)).equals(wlRi.getIdResource())) {
			// present = true;
			// break;
			// }
			//
			// }
			File file = new File(this.pathToFS, wlRi.getIdResource() + ".xml");
			if (!file.exists()) {
				wlRi = new RepoRI(RepoRI.WLRI_SCHEME + "://"
						+ this.repositoryID + "/" + getUniqueResource());
				ret = replaceUri(res, wlRi.toString());
			} else {
				ret = res;
			}
			// if (!present) {
			// wlRi = new RepoRI(RepoRI.WLRI_SCHEME + "://" + repositoryID
			// + "/" + getUniqueResource());
			// ret = replaceUri(ret, wlRi.toString());
			// }

		} else {
			wlRi = new RepoRI(RepoRI.WLRI_SCHEME + "://" + this.repositoryID
					+ "/" + getUniqueResource());
			ret = replaceUri(res, wlRi.toString());
		}

		return ret;
	}

	/**
	 * Use it to get an unique resource identifier
	 * 
	 * @return An unique resource id
	 */
	protected String getUniqueResource() {
		int max = -1;

		/*
		 * test if there is a current folder, if not creates one.
		 */
		if (this.currentFolder != null) {
			if (this.currentFolder.listFiles().length >= this.filePerFolder) {
				this.currentFolder = createNewFolder();
			} else {
				// Current folder is the right folder
			}
		} else {
			this.currentFolder = createNewFolder();
		}

		for (final File file : this.currentFolder.listFiles()) {
			final String fileName = file.getName().substring(0,
					file.getName().length() - 4);
			/*
			 * test if the file name starts with the file prefix
			 */
			if (fileName.startsWith(this.filePrefix)) {
				/*
				 * get the max value
				 */
				final int val = Integer.parseInt(fileName
						.substring(this.filePrefix.length()));
				if (val > max)
					max = val;
			}
		}
		return this.currentFolder.getName() + "/" + this.filePrefix + (max + 1);
	}

	/**
	 * Generates a new and unique folder value.
	 * 
	 * @return the new folder
	 */
	protected File createNewFolder() {
		final File folder = new File(this.pathToFS.getAbsolutePath() + "/"
				+ System.currentTimeMillis());
		if (!folder.exists() && !folder.mkdirs()) {
			LogFactory.getLog(this.getClass()).warn(
					"Unable to create folder '" + folder.getAbsolutePath()
							+ "' RuntimeException may appear later.");
		}
		return folder;
	}

	/**
	 * Replace the old uri with the new one on a resource
	 * 
	 * @param res
	 *            the resource to be changed
	 * @param newUri
	 *            the new uri
	 * @throws WebLabCheckedException
	 */
	protected Resource replaceUri(final Resource res, final String uri)
			throws WebLabCheckedException {
		final URI wlRi;
		try {
			wlRi = new URI(res.getUri());
		} catch (final URISyntaxException urise) {
			throw new WebLabCheckedException("Unvalid URI: '" + res.getUri()
					+ "'.", urise);
		}
		String oldUri = wlRi.toString();
		if (wlRi.getFragment() != null) {
			oldUri = res.getUri().substring(0, res.getUri().indexOf('#'));
		}

		final String newXmlStr;
		final ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			this.wlm.marshalResource(res, baos);
			String oldStr = baos.toString("UTF-8");
			newXmlStr = oldStr.replaceAll(oldUri, uri);
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("Unable to handle encoding.", uee);
		} finally {
			try {
				baos.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(this.getClass()).warn(
						"Unable to close stream.", ioe);
			}
		}

		ByteArrayInputStream bais;
		try {
			bais = new ByteArrayInputStream(newXmlStr.getBytes("UTF-8"));
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("Unable to handle encoding.", uee);
		}
		final Resource ret;
		try {
			ret = this.wlm.unmarshal(bais, Resource.class);
		} finally {
			try {
				bais.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(this.getClass()).warn(
						"Unable to close stream.", ioe);
			}
		}
		return ret;
	}

	// TODO Remove synchronized when WebLabMarshaller will be Thread safe.

	/**
	 * Write the resource to the file corresponding to the wlRi
	 * 
	 * @param wlRi
	 *            resource ri
	 * @param res
	 *            resource to write
	 * @throws FileNotFoundException
	 * @throws WebLabCheckedException
	 */
	protected synchronized void writeFile(final RepoRI wlRi, final Resource res)
			throws FileNotFoundException, WebLabCheckedException {
		Resource newRes;
		/*
		 * if the file already exist and if the resource to save is a resource
		 * with a fragment, we have to load the existing one and trying to
		 * update the resource.
		 */
		final File saveFile = new File(this.pathToFS.getAbsolutePath() + "/"
				+ wlRi.getIdResource() + ".xml");
		if (saveFile.exists() && wlRi.getFragment() != null) {
			/*
			 * load the file
			 */
			newRes = this.loadFile(wlRi.getIdResource() + ".xml");
			/*
			 * call getResourceInside with the last parameter : the method will
			 * find and replace the resource using the uri res.getUri()
			 */
			this.getResourceInside(newRes, res.getUri(), res);

		} else {
			newRes = res;
		}

		final FileOutputStream os = new FileOutputStream(saveFile);
		try {
			this.wlm.marshalResource(newRes, os);
		} finally {
			try {
				os.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(this.getClass()).warn(
						"Unable to close stream.", ioe);
			}
		}
	}

	/**
	 * Set the path to the folder file System (calling setPathToFS(File))
	 * Creates it if not exist
	 * 
	 * @param pathToFS
	 *            String containing the path to the folder file system
	 * @throws WebLabCheckedException
	 */
	public void setPathToFS(String pathToFS) throws WebLabCheckedException {
		if (pathToFS == null) {
			throw new WebLabCheckedException("Folder is null.");
		}
		this.setPathToFS(new File(pathToFS));
	}

	public String getRepositoryID() {
		return this.repositoryID;
	}

	public void setRepositoryID(String repositoryID) {
		this.repositoryID = repositoryID;
	}

	public String getFilePrefix() {
		return this.filePrefix;
	}

	public void setFilePrefix(String filePrefix) {
		this.filePrefix = filePrefix;
	}

	public File getPathToFS() {
		return this.pathToFS;
	}

	public static FileRepository newInstance() throws WebLabCheckedException {
		final URL fileUrl;
		final String defaultFileName = FileRepository.class.getSimpleName()
				.toLowerCase(Locale.getDefault())
				+ ".properties";
		try {
			final Enumeration<URL> urls = FileRepository.class.getClassLoader()
					.getResources(defaultFileName);
			if (urls.hasMoreElements()) {
				fileUrl = urls.nextElement();
				if (urls.hasMoreElements()) {
					LogFactory.getLog(FileRepository.class).warn(
							"Founded more than one '" + defaultFileName
									+ "' in the classpath.\nLoaded : '"
									+ fileUrl + "'.");
				}
			} else {
				throw new WebLabCheckedException("No file '" + defaultFileName
						+ "' founded in the classpath.");
			}

		} catch (final IOException ioe) {
			throw new WebLabCheckedException("No file '" + defaultFileName
					+ "' founded in the classpath.", ioe);
		}
		return FileRepository.newInstance(fileUrl);
	}

	public static FileRepository newInstance(final URL fileUrl)
			throws WebLabCheckedException {
		final Properties prop = new Properties();
		final FileRepository rep = new FileRepository();

		final String propName = FileRepository.class.getSimpleName()
				.toLowerCase(Locale.getDefault());

		InputStream fis;
		try {
			fis = fileUrl.openStream();
		} catch (final IOException fnfe) {
			throw new WebLabCheckedException("Unable to load "
					+ "properties from file: " + fileUrl, fnfe);
		}
		try {
			prop.load(fis);
		} catch (final IOException ioe) {
			throw new WebLabCheckedException("Unable to load "
					+ "properties from file: " + fileUrl, ioe);
		} finally {
			try {
				fis.close();
			} catch (final IOException ioe) {
				LogFactory.getLog(FileRepository.class).warn(
						"Unable to close stream.", ioe);
			}
		}

		final String path = prop.getProperty(propName);
		if (path == null) {
			throw new WebLabCheckedException("Property '" + propName
					+ "' not founded in '" + fileUrl + "'.");
		}

		rep.setPathToFS(path);
		try {
			rep.setFilePerFolder(Integer.parseInt(prop
					.getProperty("filePerFolder")));
		} catch (final Exception e) {
			LogFactory.getLog(FileRepository.class).warn(
					"Unable to load the property 'filePerFolder'.", e);
		}

		return rep;
	}

	public int getFilePerFolder() {
		return this.filePerFolder;
	}

	public void setFilePerFolder(int filePerFolder) {
		this.filePerFolder = filePerFolder;
	}

	public File getCurrentFolder() {
		return this.currentFolder;
	}

	public void setCurrentFolder(final File currentFolder) {
		this.currentFolder = currentFolder;
		if (!currentFolder.exists() && !currentFolder.mkdirs()) {
			LogFactory.getLog(this.getClass()).warn(
					"Unable to create folder. "
							+ "RuntimeException may appears.");
		}
	}

}
