/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.repository.file;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabRISyntaxException;

public class RepoRI {


	public final static String WLRI_REGEX = "^weblab:(//([^/?#\\n]+))+/(([^/?#\\n][^?#\\n]+)(\\?([^#\\n]+))?)+(#([^\\n]+))?";


	public final static String WLRI_SCHEME = "weblab";


	private URI uri;


	private String idReference;


	private String idResource;


	private final String fragment;


	/**
	 * Create a <code>WebLabRI</code> with tests.
	 * 
	 * @param uriStr
	 *            a string representation of the <code>WebLabRI</code>
	 */
	public RepoRI(final String uriStr) throws WebLabCheckedException {
		super();

		// first check: a valid WebLabRI must be a valid URI
		try {
			this.uri = new URI(uriStr);
		} catch (final URISyntaxException urise) {
			throw new WebLabCheckedException("Unable to create URI from: '" + uriStr + "'.", urise);
		}

		// second: scheme must be 'weblab'
		if (!this.uri.getScheme().equals(RepoRI.WLRI_SCHEME)) {
			throw new WebLabCheckedException("Scheme must be weblab, founded " + this.uri.getScheme());
		}

		final Pattern p = Pattern.compile(RepoRI.WLRI_REGEX);
		final Matcher m = p.matcher(uriStr);

		// third: URI must be compliant with WebLabRI regex
		if (m.matches()) {
			this.idReference = m.group(2);
			this.idResource = m.group(3);
		} else {
			throw new WebLabCheckedException("Invalid WebLabRI: '" + uriStr + "'.");
		}

		// May be null if no fragment
		this.fragment = this.uri.getFragment();
	}


	/**
	 * Complete a <code>WebLabRI</code> fragment with a <code>String</code>
	 * 
	 * @param fragmentToAdd
	 *            Added to the <code>WebLabRI</code> fragment
	 */
	public void addFragment(final String fragmentToAdd) {
		if (this.uri.getFragment() == null) {
			try {
				this.uri = new URI(this.uri.toString() + "#" + fragmentToAdd);
			} catch (final URISyntaxException urise) {
				throw new WebLabRISyntaxException(urise);
			}
		} else {
			try {
				this.uri = new URI(this.uri.toString() + "-" + fragmentToAdd);
			} catch (final URISyntaxException urise) {
				throw new WebLabRISyntaxException(urise);
			}
		}
	}


	/**
	 * @return the <code>idReference</code>
	 */
	public String getIdReference() {
		return this.idReference;
	}


	/**
	 * @return the <code>idReference</code>
	 */
	public String getIdResource() {
		return this.idResource;
	}


	/**
	 * @return the <code>fragment</code>
	 */
	public String getFragment() {
		return this.fragment;
	}


	@Override
	public String toString() {
		return this.uri.toString();
	}

}
