/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.repository.file;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Resource;


/**
 * Simple implementation of a file repository Save each resource to a File.
 * 
 * @author jdoucy, ymombrun - Cassidian
 */
public class FileRepository {


	protected final File pathToFS;


	protected final String filePrefix;


	protected final String repositoryID;


	protected final int filePerFolder;


	protected final Log log;


	protected File currentFolder;


	protected final WebLabMarshaller wlm = new WebLabMarshaller();


	public FileRepository(final String pathToTheRepository, final int filePerFolder, final String repositoryID, final String filePrefix) throws IOException {
		this.log = LogFactory.getLog(this.getClass());
		this.filePrefix = filePrefix;
		this.filePerFolder = filePerFolder;
		this.repositoryID = repositoryID;
		this.pathToFS = this.initPathToFS(pathToTheRepository);
		this.createNewSubFolder();
		this.log.debug("Repository initialised on " + this.pathToFS.getAbsolutePath() + ".");
	}




	/**
	 * @param uri
	 *            The URI of the resource to retrieve
	 * @return The Resource
	 * @throws WebLabCheckedException
	 *             If the resource is not found
	 * @throws FileNotFoundException
	 */
	public Resource getResource(final String uri) throws WebLabCheckedException, FileNotFoundException {
		final RepoRI wlRi = new RepoRI(uri);
		final Resource res;
		try {
			res = this.loadFile(wlRi.getIdResource() + ".xml");
		} catch (final FileNotFoundException fnfe) {
			throw new FileNotFoundException("Uri '" + uri + "' is not present in the repository.");
		}
		final Resource result = this.getResourceInside(res, uri);
		if (result == null) {
			throw new FileNotFoundException("Uri '" + uri + "' is not present in the repository.");
		}
		return result;
	}


	/**
	 * @param resource
	 *            The resource to save
	 * @return The new (might be identical if the resource has already been save in this repo)
	 * @throws WebLabCheckedException
	 */
	public String saveResource(final Resource resource) throws WebLabCheckedException {
		final Resource res = this.changeUri(resource);

		try {
			this.writeFile(new RepoRI(res.getUri()), res);
		} catch (final FileNotFoundException fnfe) {
			throw new WebLabCheckedException("Unable to write resource: " + resource.getUri() + " into resource: " + res.getUri() + ".", fnfe);
		}

		return res.getUri();
	}


	/**
	 * Recursive route which return resource corresponding to the uri
	 * 
	 * @param res
	 *            main resource
	 * @param uri
	 *            uri to find
	 * @return res corresponding to the uri
	 */
	protected Resource getResourceInside(final Resource res, final String uri) throws WebLabCheckedException {
		return this.getResourceInside(res, uri, null);
	}


	/**
	 * recursive route which replace newRes in res
	 * 
	 * @param res
	 *            main resource
	 * @param uri
	 *            uri to find (the same as in newRes)
	 * @param newRes
	 *            new resource which going to replace the funded one
	 * @return res corresponding to the uri
	 */
	protected Resource getResourceInside(final Resource res, final String uri, final Resource newRes) throws WebLabCheckedException {
		return this.getResourceInside(res, uri, newRes, null);
	}


	protected Resource getResourceInside(final Resource res, final String uri, final Resource newRes, final Resource father) throws WebLabCheckedException {

		/*
		 * if it's the current return it
		 */
		if (res.getUri().equals(uri)) {
			if (newRes != null) {
				this.replaceResource(father, res, newRes);
			}
			return res;
		}

		/*
		 * recursive call for annotations tests
		 */
		final List<Annotation> listAnnotCopy = new ArrayList<Annotation>(res.getAnnotation());
		Resource ret = null;
		final Iterator<Annotation> itAnnot = listAnnotCopy.iterator();
		while ((ret == null) && itAnnot.hasNext()) {
			ret = this.getResourceInside(itAnnot.next(), uri, newRes, res);
		}

		if (ret != null) {
			return ret;
		}

		/*
		 * recursive call for mediaUnits tests
		 */
		if (res instanceof Document) {
			final Document document = (Document) res;
			final List<MediaUnit> listMediaUnitCopy = new ArrayList<MediaUnit>(document.getMediaUnit());
			final Iterator<MediaUnit> itMed = listMediaUnitCopy.iterator();
			while ((ret == null) && itMed.hasNext()) {
				ret = this.getResourceInside(itMed.next(), uri, newRes, res);
			}

			if (ret != null) {
				return ret;
			}
		}

		return ret;
	}


	protected void replaceResource(final Resource father, final Resource oldRes, final Resource newRes) throws WebLabCheckedException {

		/*
		 * try to replace the found one in the list
		 */
		if ((newRes instanceof MediaUnit) && (father instanceof Document)) {
			final MediaUnit newMed = (MediaUnit) newRes;
			final Document fatherDoc = (Document) father;
			final int index = fatherDoc.getMediaUnit().indexOf(oldRes);
			fatherDoc.getMediaUnit().remove(oldRes);
			fatherDoc.getMediaUnit().add(index, newMed);
		} else if (newRes instanceof Annotation) {
			final Annotation newAnnot = (Annotation) newRes;
			final int index = father.getAnnotation().indexOf(oldRes);
			father.getAnnotation().remove(oldRes);
			father.getAnnotation().add(index, newAnnot);
		} else {
			throw new WebLabCheckedException("In the document repository, the uri '" + newRes.getUri() + "' is a '" + oldRes.getClass()
					+ "' and is not replacable by a '" + newRes.getClass() + "'.");
		}
	}


	/**
	 * Load the resource corresponding to the file name
	 * 
	 * @param fileName
	 * @return the loaded resource
	 * @throws WebLabCheckedException
	 * @throws FileNotFoundException
	 */
	protected synchronized Resource loadFile(final String fileName) throws WebLabCheckedException, FileNotFoundException {
		final File file = new File(this.pathToFS.getAbsoluteFile(), fileName);
		if (!file.exists()) {
			throw new FileNotFoundException(file.getAbsolutePath() + " not found.");
		}
		return this.wlm.unmarshal(file, Resource.class);
	}


	/**
	 * Check if the resource is already present if not create a unique wlri and change it in the resource
	 * 
	 * @param res
	 *            resource
	 * @return new unique wlri
	 * @throws WebLabCheckedException
	 */
	protected Resource changeUri(final Resource res) throws WebLabCheckedException {

		RepoRI wlRi;
		if (RepoRI.checkValidity(res.getUri())) {
			try {
				wlRi = new RepoRI(res.getUri());
			} catch (final WebLabCheckedException wlce) {
				LogFactory.getLog(this.getClass()).warn("Unable to load a valid WLRI: '" + res.getUri() + "'.", wlce);
				wlRi = null;
			}
		} else {
			wlRi = null;
		}


		/*
		 * first, is the isRef the current one
		 */
		final Resource ret;
		if ((wlRi != null) && this.repositoryID.equals(wlRi.getIdReference())) {
			final File file = new File(this.pathToFS, wlRi.getIdResource() + ".xml");
			if (!file.exists()) {
				wlRi = new RepoRI(RepoRI.WLRI_SCHEME + "://" + this.repositoryID + "/" + this.getUniqueResource());
				ret = this.replaceUri(res, wlRi.toString());
			} else {
				ret = res;
			}
		} else {
			wlRi = new RepoRI(RepoRI.WLRI_SCHEME + "://" + this.repositoryID + "/" + this.getUniqueResource());
			ret = this.replaceUri(res, wlRi.toString());
		}

		return ret;
	}


	/**
	 * Use it to get an unique resource identifier
	 * 
	 * @return An unique resource id
	 */
	protected String getUniqueResource() {
		int max = -1;

		/*
		 * test if there is a current folder, if not creates one.
		 */
		if (this.currentFolder != null) {
			if (this.currentFolder.listFiles().length >= this.filePerFolder) {
				this.createNewSubFolder();
			}
			// Current folder is the right folder
		} else {
			this.createNewSubFolder();
		}

		for (final File file : this.currentFolder.listFiles()) {
			final String fileName = file.getName().substring(0, file.getName().length() - 4);
			/*
			 * test if the file name starts with the file prefix
			 */
			if (fileName.startsWith(this.filePrefix)) {
				/*
				 * get the max value
				 */
				final int val = Integer.parseInt(fileName.substring(this.filePrefix.length()));
				if (val > max) {
					max = val;
				}
			}
		}
		return this.currentFolder.getName() + "/" + this.filePrefix + (max + 1);
	}



	/**
	 * Replace the old uri with the new one on a resource
	 * 
	 * @param res
	 *            the resource to be changed
	 * @param newUri
	 *            the new uri
	 * @throws WebLabCheckedException
	 */
	protected Resource replaceUri(final Resource res, final String uri) throws WebLabCheckedException {
		final URI wlRi;
		try {
			wlRi = new URI(res.getUri());
		} catch (final URISyntaxException urise) {
			throw new WebLabCheckedException("Unvalid URI: '" + res.getUri() + "'.", urise);
		}
		String oldUri = wlRi.toString();
		if (wlRi.getFragment() != null) {
			oldUri = res.getUri().substring(0, res.getUri().indexOf('#'));
		}

		Resource ret = this.stringReplaceInResource(res, uri, oldUri);

		if (wlRi.getFragment() != null) {
			// This is needed to fix #WEBLAB-417
			final String topRetUri = ret.getUri();
			ret = this.stringReplaceInResource(ret, "\"" + uri + "\"", "\"" + topRetUri + "\"");
			ret = this.stringReplaceInResource(ret, uri + "#", topRetUri);
		}

		return ret;
	}




	/**
	 * @param resource
	 *            The resource to be marshaled and unmarshashaled
	 * @param target
	 *            The String to be search in the marshaled resource
	 * @param replacement
	 *            The replacement for each target found
	 * @return The resource with target replaced by replacement
	 * @throws WebLabCheckedException
	 */
	private Resource stringReplaceInResource(final Resource resource, final String replacement, final String target) throws WebLabCheckedException {
		final String newXmlStr;
		final ByteArrayOutputStream baos = new ByteArrayOutputStream();
		try {
			this.wlm.marshalResource(resource, baos);
			final String oldStr = baos.toString("UTF-8");
			newXmlStr = oldStr.replace(target, replacement);
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("Unable to handle encoding.", uee);
		} finally {
			IOUtils.closeQuietly(baos);
		}

		final ByteArrayInputStream bais;
		try {
			bais = new ByteArrayInputStream(newXmlStr.getBytes("UTF-8"));
		} catch (final UnsupportedEncodingException uee) {
			throw new WebLabCheckedException("Unable to handle encoding.", uee);
		}
		final Resource ret;
		try {
			ret = this.wlm.unmarshal(bais, Resource.class);
		} finally {
			IOUtils.closeQuietly(bais);
		}
		return ret;
	}


	/**
	 * Write the resource to the file corresponding to the wlRi
	 * 
	 * @param wlRi
	 *            resource ri
	 * @param res
	 *            resource to write
	 * @throws FileNotFoundException
	 * @throws WebLabCheckedException
	 */
	protected synchronized void writeFile(final RepoRI wlRi, final Resource res) throws FileNotFoundException, WebLabCheckedException {
		Resource newRes;
		/*
		 * if the file already exist and if the resource to save is a resource
		 * with a fragment, we have to load the existing one and trying to
		 * update the resource.
		 */
		final File saveFile = new File(this.pathToFS.getAbsolutePath(), wlRi.getIdResource() + ".xml");
		if (saveFile.exists() && (wlRi.getFragment() != null)) {
			/*
			 * load the file
			 */
			newRes = this.loadFile(wlRi.getIdResource() + ".xml");
			/*
			 * call getResourceInside with the last parameter : the method will
			 * find and replace the resource using the uri res.getUri()
			 */
			this.getResourceInside(newRes, res.getUri(), res);

		} else {
			newRes = res;
		}

		this.wlm.marshalResource(newRes, saveFile);
	}



	/**
	 * @param pathToTheRepository
	 *            The path to the root folder of the repository
	 * @return A File on this folder
	 * @throws IOException
	 *             If the folder cannot be created
	 */
	protected File initPathToFS(final String pathToTheRepository) throws IOException {
		if (pathToTheRepository == null) {
			throw new IOException("Unable to create a folder with null as name.");
		}

		final File file = new File(pathToTheRepository);
		if (!file.isAbsolute()) {
			this.log.warn("The path to the repository is not absolute. This might lead to problem if the server is not started from the same location.");
		}

		// Check existence and create folder hierarchy if needed
		if (!file.exists() && !file.mkdirs()) {
			throw new IOException("Unable to create folder " + file.getAbsolutePath());
		}

		if (!file.isDirectory()) {
			throw new IOException("Unable to create folder " + file.getAbsolutePath() + " since a file with the same name exists.");
		}

		return file;
	}


	/**
	 * Generates a new and unique folder value.
	 * 
	 * @return the new folder
	 */
	protected void createNewSubFolder() {
		final File folder = new File(this.pathToFS.getAbsolutePath(), "" + System.currentTimeMillis());
		this.setCurrentFolder(folder);
	}


	protected void setCurrentFolder(final File currentFolder) {
		this.currentFolder = currentFolder;
		if (!currentFolder.exists() && !currentFolder.mkdirs()) {
			LogFactory.getLog(this.getClass()).warn("Unable to create folder. " + "RuntimeException may appears.");
		}
	}

}
