/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.repository.file.impl;

import java.io.FileNotFoundException;
import java.io.IOException;

import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ResourceContainer;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceArgs;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceReturn;
import org.ow2.weblab.core.services.resourcecontainer.SaveResourceArgs;
import org.ow2.weblab.core.services.resourcecontainer.SaveResourceReturn;
import org.ow2.weblab.repository.file.FileRepository;

/**
 * This class is service in charge of storing content in it.
 * It will change the URIs of the received resources if they are not known.
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.ResourceContainer")
public class FileRepositoryService implements ResourceContainer {


	public static final String DEFAULT_REPO_ID = "fileRepository";


	public static final String DEFAULT_PATH = "repository";


	public final static int DEFAULT_FILE_PER_FOLDER = 1000;


	public static final String DEFAULT_FILE_PREFIX = "res_";


	protected final FileRepository repo;


	protected final Log log;



	/**
	 * The default constructor
	 */
	public FileRepositoryService() {
		this(FileRepositoryService.DEFAULT_PATH);
	}


	/**
	 * @param pathToTheRepository
	 *            The path to the root folder of the repository
	 */
	public FileRepositoryService(final String pathToTheRepository) {
		this(pathToTheRepository, FileRepositoryService.DEFAULT_FILE_PER_FOLDER);
	}


	/**
	 * @param pathToTheRepository
	 *            The path to the root folder of the repository
	 * @param numberOfFilePerFolder
	 *            The number of file per sub folder in the repository
	 */
	public FileRepositoryService(final String pathToTheRepository, final int numberOfFilePerFolder) {
		this(pathToTheRepository, numberOfFilePerFolder, FileRepositoryService.DEFAULT_REPO_ID, FileRepositoryService.DEFAULT_FILE_PREFIX);
	}


	/**
	 * @param pathToTheRepository
	 *            The path to the root folder of the repository
	 * @param numberOfFilePerFolder
	 *            The number of file per sub folder in the repository
	 * @param repoId
	 *            The name of the repository used in the URIs
	 * @param filePrefix
	 *            The prefix to be used in file names
	 */
	public FileRepositoryService(final String pathToTheRepository, final int numberOfFilePerFolder, final String repoId, final String filePrefix) {
		super();
		this.log = LogFactory.getLog(this.getClass());
		try {
			this.repo = new FileRepository(pathToTheRepository, numberOfFilePerFolder, repoId, filePrefix);
		} catch (final IOException ioe) {
			this.log.fatal("Service cannot be started due to an IOException at initialisation.", ioe);
			throw new WebLabUncheckedException(ioe);
		}
	}


	@Override
	public LoadResourceReturn loadResource(final LoadResourceArgs args) throws InvalidParameterException, UnexpectedException {
		if (args == null) {
			throw new InvalidParameterException("LoadResourceArgs was null.", "LoadResourceArgs was null.");
		}

		final String resId = args.getResourceId();
		if ((resId == null) || "".equals(resId.trim())) {
			throw new InvalidParameterException("ResourceId in LoadResourceArgs was null or empty.", "ResourceId in LoadResourceArgs was null or empty.");
		}

		Resource res;
		try {
			res = this.repo.getResource(resId);
		} catch (final FileNotFoundException fnfe) {
			this.log.error("Unable to get resource from URI " + resId + ".", fnfe);
			throw new InvalidParameterException("Unable to get resource from URI " + resId + ".", "Unable to get resource from URI " + resId + ".", fnfe);
		} catch (final WebLabCheckedException wlce) {
			this.log.error("Unable to get resource from URI " + resId + ".", wlce);
			throw new UnexpectedException("Unable to get resource from URI " + resId + ".", "Unable to get resource from URI " + resId + ".", wlce);
		}

		final LoadResourceReturn ret = new LoadResourceReturn();
		ret.setResource(res);
		return ret;
	}


	@Override
	public SaveResourceReturn saveResource(final SaveResourceArgs args) throws InvalidParameterException, UnexpectedException {

		if (args == null) {
			throw new InvalidParameterException("SaveResourceArgs was null.", "SaveResourceArgs was null.");
		}

		final Resource res = args.getResource();
		if ((res == null) || (res.getUri() == null)) {
			throw new InvalidParameterException("Resource SaveResourceException was null or without URI.",
					"Resource SaveResourceException was null or without URI.");
		}

		String resId;
		try {
			resId = this.repo.saveResource(res);
		} catch (final WebLabCheckedException wlce) {
			final String msg = "Unable to save resource with URI " + res.getUri() + ".";
			this.log.error(msg, wlce);
			throw new UnexpectedException(msg, msg, wlce);
		}

		final SaveResourceReturn ret = new SaveResourceReturn();
		ret.setResourceId(resId);
		return ret;
	}

}
