/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.services.iterator;

import java.io.File;
import java.util.Iterator;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.EmptyQueueException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.QueueManager;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.queuemanager.NextResourceArgs;
import org.ow2.weblab.core.services.queuemanager.NextResourceReturn;
import org.ow2.weblab.services.iterator.messages.Keys;
import org.ow2.weblab.services.iterator.messages.Messages;


/**
 * This class is a the QueueManager part of the FolderResource service.
 * It enable to crawl a folder (given through the configure method of the Configurable interface) or through the default path of the configuration.
 * Some filters are applied to the folder while crawling. They are created using the configuration in ConfigurationBean set in the constructor method.
 * Each of the file MUST be valid WebLab XML resources that will be returned each time the nextResourceArgs is called.
 * When every files have been crawled, an EmptyQueue exception is thrown.
 * 
 * @author ymombrun
 * @date 2011-08-17
 * @see ConfigurationBean
 */
public class FolderResourceQM implements QueueManager {


	/**
	 * The logger used inside
	 */
	private final Log logger;


	/**
	 * The bean that contains the configuration of the service.
	 */
	private final ConfigurationBean configuration;


	/**
	 * The mapping between Configurable and QueueManager.
	 */
	private final InterfacesMappingSingleton mapping;


	/**
	 * The (un)marshaller used to read
	 */
	private final WebLabMarshaller marshaller;



	/**
	 * The constructor for the realisation of the QueueManager interface for a FolderResource service.
	 * 
	 * @param configuration
	 *            A holder for the configuration of this service.
	 * @throws UnexpectedException
	 *             Cannot be launched here (the method called inside throw it only if default path is <code>null</code>. But this method is not called if
	 *             default path is <code>null</code>.)
	 * @throws AccessDeniedException
	 *             If the path represented by default path does not exists, is not a folder or is not readable.
	 */
	public FolderResourceQM(final ConfigurationBean configuration) throws UnexpectedException, AccessDeniedException {
		this.logger = LogFactory.getLog(this.getClass());
		this.configuration = configuration;
		this.mapping = InterfacesMappingSingleton.getInstance();
		if (this.configuration.getDefaultPath() != null) {
			this.configureWithDefault("");
		}
		this.marshaller = new WebLabMarshaller();
		this.logger.info(Messages.getString(Keys.QM_STARTED, this.configuration.toString()));
	}



	@Override
	public NextResourceReturn nextResource(final NextResourceArgs args) throws EmptyQueueException, UnexpectedException, InvalidParameterException,
			ServiceNotConfiguredException, AccessDeniedException {
		this.logger.debug("Early start of nextResource Method");

		// Get usageContext from args and checks that it has been configured.
		final String usageContext = this.checkNextResourceArgs(args);

		this.logger.info(Messages.getString(Keys.QM_NEXT_RESOURCE_CALLED, usageContext));

		// Retrieve the next file in the configuration
		final File f = this.getFile(usageContext);

		this.logger.debug("Try unmarshall file '" + f.getAbsolutePath() + "' for usageContext '" + usageContext + "'.");

		// Unmarshal the resource
		final Resource res;
		try {
			res = this.marshaller.unmarshal(f, Resource.class);
		} catch (final WebLabCheckedException wlce) {
			throw new UnexpectedException(Messages.getString(Keys.QM_UNMARSHAL_ERROR, usageContext, f.getPath()), wlce);
		}

		// Log it only in trace mode
		if (this.logger.isTraceEnabled()) {
			try {
				this.logger.trace(ResourceUtil.saveToXMLString(res));
			} catch (final WebLabCheckedException wlce) {
				this.logger.warn(Messages.getString(Keys.QM_MARSHAL_ERROR, usageContext, f.getPath(), res.getUri()), wlce);
			}
		}

		// Remove file if needed.
		if (this.configuration.isRemoveFiles()) {
			this.logger.debug("Try remove file '" + f.getAbsolutePath() + "' for usageContext '" + usageContext + "'.");
			FileUtils.deleteQuietly(f);
		}

		// Return the unmarshaled resource
		final NextResourceReturn nrr = new NextResourceReturn();
		nrr.setResource(res);

		return nrr;
	}


	/**
	 * @param usageContext
	 *            The usageContext used to access the iterator in the configuration.
	 * @return The next file in the iterator
	 * @throws ServiceNotConfiguredException
	 *             If no iterator can be found in the mapping. But this should never appear since it's check in {@link #checkNextResourceArgs(NextResourceArgs)}
	 * @throws EmptyQueueException
	 *             If the iterator is empty.
	 * @throws UnexpectedException
	 *             If the iterator returns an object that is not a file. This should never appear.
	 * @throws AccessDeniedException
	 *             If the file return by the iterator is not readable, does not exist or is not a file.
	 */
	private File getFile(final String usageContext) throws ServiceNotConfiguredException, EmptyQueueException, UnexpectedException, AccessDeniedException {
		final File f;
		// A synchronised block on the singleton to prevent from multi-threaded calls.
		synchronized (this.mapping) {
			final Iterator<?> it = this.mapping.getIterator(usageContext);
			if (it == null) {
				// This should never appear since in #checkNextResourceArgs we check that usageContext is configured
				throw new ServiceNotConfiguredException(Messages.getString(Keys.QM_NOT_CONFIGURED, usageContext));
			}

			if (!it.hasNext()) {
				this.logger.debug("No more resource in iterator for usageContext '" + usageContext + "'.");
				throw new EmptyQueueException(Messages.getString(Keys.QM_EMPTY, usageContext));
			}

			final Object o = it.next();
			if (!(o instanceof File)) {
				throw new UnexpectedException(Messages.getString(Keys.QM_NOT_A_FILE, usageContext, o));
			}
			f = (File) o;
		}

		if (!f.exists()) {
			throw new AccessDeniedException(Messages.getString(Keys.QM_FILE_NOT_EXIST, usageContext, f.getPath()));
		}
		if (!f.isFile()) {
			throw new AccessDeniedException(Messages.getString(Keys.QM_FILE_NOT_FILE, usageContext, f.getPath()));
		}
		if (!f.canRead()) {
			throw new AccessDeniedException(Messages.getString(Keys.QM_FILE_NOT_READABLE, usageContext, f.getPath()));
		}

		return f;
	}



	/**
	 * Check that <code>args</code> is suitable for the call to {@link #nextResource(NextResourceArgs)}.
	 * It checks if the singleton mapping contains a configuration of the usageContext in args. If not (and if defined in the configuration), try configuring it
	 * with default configuration.
	 * 
	 * @param args
	 *            The NextResourceArgs to test validity and configuration.
	 * @return The <code>UsageContext</code> in <code>args</code> or the empty String if <code>null</code>.
	 * @throws InvalidParameterException
	 *             If <code>args</code> is <code>null</code>.
	 * @throws ServiceNotConfiguredException
	 *             If the <code>usageContext</code> has not been configured and if the useDefaultPathForNotConfigured property is false.
	 * @throws AccessDeniedException
	 *             If default path does not exist is not readable or is not a folder. And the usageContext was not configured (and the configuration ask to
	 *             configure unknown usageContext with default path)
	 * @throws UnexpectedException
	 *             If default path is <code>null</code> and the usageContext was not configured (and the configuration ask to configure unknown usageContext
	 *             with default path)
	 */
	private String checkNextResourceArgs(final NextResourceArgs args) throws InvalidParameterException, ServiceNotConfiguredException, UnexpectedException,
			AccessDeniedException {

		if (args == null) {
			throw new InvalidParameterException(Messages.getString(Keys.QM_ARGS_NULL));
		}

		String uc = args.getUsageContext();
		if (uc == null) {
			this.logger.debug("UsageContext was null, use the empty string.");
			uc = "";
		}

		if (!this.mapping.isConfigured(uc) && this.configuration.isUseDefaultPathForNotConfigured()) {
			this.configureWithDefault(uc);
		} else if (!this.mapping.isConfigured(uc)) {
			throw new ServiceNotConfiguredException(Messages.getString(Keys.QM_NOT_CONFIGURED, uc));
		}

		return uc;
	}



	/**
	 * @param uc
	 *            The usageContext for the configuration
	 * @throws UnexpectedException
	 *             If the default path in the configuration is <code>null</code>.
	 * @throws AccessDeniedException
	 *             If the default path is not readable, does not exist or is not a folder.
	 */
	private synchronized void configureWithDefault(final String uc) throws UnexpectedException, AccessDeniedException {
		final String folder = this.configuration.getDefaultFolder();
		if (folder == null) {
			throw new UnexpectedException(Messages.getString(Keys.QM_PATH_NULL, uc));
		}
		final File f = FolderResourceConf.checkFolder(folder, uc);
		final Iterator<?> it = FileUtils.iterateFiles(f, this.configuration.getFileFilter(), this.configuration.getDirectoryFilter());
		this.mapping.addToMap(uc, it);
	}

}
