/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.services.iterator;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.services.Configurable;
import org.ow2.weblab.core.services.QueueManager;
import org.ow2.weblab.services.iterator.messages.Keys;
import org.ow2.weblab.services.iterator.messages.Messages;


/**
 * This class is a singleton in charge of the mapping between the two interfaces of the service ({@link Configurable} and {@link QueueManager}).
 * 
 * @author ymombrun
 * @date 2011-08-17
 */
public class InterfacesMappingSingleton {


	/**
	 * The singleton used to map between the two services (Configurable and QueueManager)
	 */
	private static InterfacesMappingSingleton SINGLETON = null;


	/**
	 * Get the existing instance of MappingSingleton or initialise it.
	 * 
	 * @return The singleton
	 */
	public static synchronized InterfacesMappingSingleton getInstance() {
		if (InterfacesMappingSingleton.SINGLETON == null) {
			InterfacesMappingSingleton.SINGLETON = new InterfacesMappingSingleton();
		}
		return InterfacesMappingSingleton.SINGLETON;
	}


	/**
	 * The mapping between usageContexts and associated file iterators.
	 * 
	 * (Note that due to a limitation in commons-io, iterators are not parametrised with file, even if they should never return anything else than File).
	 */
	private final Map<String, Iterator<?>> iteratorByUsageContext;


	/**
	 * The logger used in the class
	 */
	private final Log logger;


	/**
	 * The default constructor that should only be called once, by the getInstance method
	 */
	private InterfacesMappingSingleton() {
		this.logger = LogFactory.getLog(this.getClass());
		this.iteratorByUsageContext = Collections.synchronizedMap(new HashMap<String, Iterator<?>>());
		this.logger.info(Messages.getString(Keys.IMS_INITIALISED));
	}


	/**
	 * Add to the configuration map this mapping between <code>usageContext</code> and <code>fileIterator</code>.
	 * 
	 * @param usageContext
	 *            A non-null <code>UsageContext</code> used for the configuration.
	 * @param fileIterator
	 *            The file iterator generated by commons-io, to be associated with <code>usageContext</code>.
	 * @return The previous value associated with <code>usageContext</code>, or <code>null</code> if there was no mapping for <code>usageContext</code>.
	 */
	public synchronized Iterator<?> addToMap(final String usageContext, final Iterator<?> fileIterator) {
		this.logger.debug("Add iterator for usageContext '" + usageContext + "'.");
		return this.iteratorByUsageContext.put(usageContext, fileIterator);
	}


	/**
	 * Removes every mapping from the configuration map. Bug before, for every mapping, it consumes their iterator till their end to eases their collection by
	 * the garbage collector.
	 */
	public synchronized void clearMap() {
		this.logger.debug("Start emptying the map.");
		for (final Entry<String, Iterator<?>> entry : this.iteratorByUsageContext.entrySet()) {
			// Consume each iterator till its end to empty memory
			final Iterator<?> it = entry.getValue();
			while (it.hasNext()) {
				it.next();
			}
		}
		this.iteratorByUsageContext.clear();
		this.logger.debug("Map emptied.");
	}


	/**
	 * Get the iterator associated to this usageContext or <code>null</code> if not configured.
	 * 
	 * @param usageContext
	 *            A non-null <code>UsageContext</code> previously configured.
	 * @return The iterator configured with the <code>usageContext</code> or <code>null</code> if not configured.
	 */
	public synchronized Iterator<?> getIterator(final String usageContext) {
		this.logger.debug("Retrieve iterator for usageContext '" + usageContext + "'.");
		return this.iteratorByUsageContext.get(usageContext);
	}


	/**
	 * Check if the usageContext in parameter has been configured.
	 * 
	 * @param usageContext
	 *            The <code>String</code> to test existence in the configuration map.
	 * @return Whether <code>usageContext</code> is in the map.
	 */
	public synchronized boolean isConfigured(final String usageContext) {
		return this.iteratorByUsageContext.containsKey(usageContext);
	}


	/**
	 * Removes <code>usageContext</code> from the configuration map. If a mapping was defined, it consumes the iterator till it's end to eases its collection by
	 * the garbage collector.
	 * 
	 * @param usageContext
	 *            A non-null <code>UsageContext</code> to remove the configuration.
	 */
	public synchronized void removeConfiguration(final String usageContext) {
		if (this.iteratorByUsageContext.containsKey(usageContext)) {
			this.logger.debug("Remove iterator for usageContext '" + usageContext + "'.");
			final Iterator<?> it = this.iteratorByUsageContext.remove(usageContext);
			// Consume the iterator till its end to empty memory
			while (it.hasNext()) {
				it.next();
			}
		} else {
			this.logger.debug("Nothing to remove for usageContext '" + usageContext + "'.");
		}
	}

}
