/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.iterator;

import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

import javax.jws.WebService;

import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Configurable;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.configurable.ConfigureArgs;
import org.ow2.weblab.core.services.configurable.ConfigureReturn;
import org.ow2.weblab.core.services.configurable.ResetConfigurationArgs;
import org.ow2.weblab.core.services.configurable.ResetConfigurationReturn;
import org.ow2.weblab.services.iterator.messages.Keys;
import org.ow2.weblab.services.iterator.messages.Messages;


/**
 * This class is a the Configurable part of the FolderResource service.
 * It enable to configure the folder to be crawled by usageContext or to reset configurations.
 * 
 * The PoK of configuration:
 * <ul>
 * <li>must contain one statement from the URI of the service to the URI of configuration instance with hasConfiguration as predicate;</li>
 * <li>should contain one statement defining that this configuration is of type ConfigurationType;</li>
 * <li>should contain one statement linking this configuration with the usageContext it configures;</li>
 * <li>must contain one statement defining the folderToCrawl of this configuration instance.</li>
 * </ul>
 * 
 * @see InterfacesMappingSingleton
 * @author ymombrun
 * @date 2011-08-18
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.Configurable")
public class FolderResourceConf implements Configurable {


	/**
	 * The bean that contains the configuration of the service.
	 */
	protected final ConfigurationBean configuration;


	/**
	 * The logger used inside.
	 */
	protected final Log logger;


	/**
	 * The class that maps Configurable with QueueManager.
	 */
	protected final InterfacesMappingSingleton mapping;


	/**
	 * The constructor that takes as parameter the same bean than the queue manager to be associated with.
	 * 
	 * @param configuration
	 *            The configuration used here.
	 */
	public FolderResourceConf(final ConfigurationBean configuration) {
		this.logger = LogFactory.getLog(this.getClass());
		this.mapping = InterfacesMappingSingleton.getInstance();
		this.configuration = configuration;
		this.logger.info(Messages.getString(Keys.C_STARTED));
	}


	@Override
	public ConfigureReturn configure(final ConfigureArgs args) throws InvalidParameterException, AccessDeniedException, UnexpectedException {
		if (args == null) {
			throw new InvalidParameterException(Messages.getString(Keys.C_CONF_ARGS_NULL));
		}
		String usageContext = args.getUsageContext();

		final PieceOfKnowledge conf = args.getConfiguration();
		if (conf == null) {
			throw new InvalidParameterException(Messages.getString(Keys.C_CONF_NULL, usageContext));
		}

		if (usageContext == null) {
			usageContext = "";
		}

		final String folderToCrawl = this.readConfigPoK(usageContext, conf);
		final File file = FolderResourceConf.checkFolder(folderToCrawl, usageContext);
		final Iterator<?> it = FileUtils.iterateFiles(file, this.configuration.getFileFilter(), this.configuration.getDirectoryFilter());

		synchronized (this.mapping) {
			if (this.mapping.isConfigured(usageContext)) {
				this.logger.warn(Messages.getString(Keys.C_CONF_EXIST, usageContext));
				this.mapping.removeConfiguration(usageContext);
			}
			this.logger.info(Messages.getString(Keys.C_CONF, usageContext, file.getAbsolutePath()));
			this.mapping.addToMap(usageContext, it);
		}

		return new ConfigureReturn();
	}


	@Override
	public ResetConfigurationReturn resetConfiguration(final ResetConfigurationArgs args) throws InvalidParameterException, UnexpectedException {
		if (args == null) {
			throw new InvalidParameterException(Messages.getString(Keys.C_RESET_CONF_NULL));
		}
		final String usageContext = args.getUsageContext();
		if (usageContext == null) {
			this.logger.info(Messages.getString(Keys.C_CLEAR_CONF));
			this.mapping.clearMap();
		} else {
			this.logger.info(Messages.getString(Keys.C_RESET_UC, usageContext));
			this.mapping.removeConfiguration(usageContext);
		}
		return new ResetConfigurationReturn();
	}


	/**
	 * Reads the PieceOfKnowledge in parameter to retrieve path to the folder to be crawled.
	 * 
	 * @param usageContext
	 *            The usageContext send as parameter of the configuration.
	 * @param conf
	 *            The PoK that should contain the configuration.
	 * @return The path to the folder to crawl (not tested).
	 * @throws InvalidParameterException
	 *             If no configuration has been found (linked with the service Uri) or if no folder to crawl is provided in this configuration.
	 */
	public String readConfigPoK(final String usageContext, final PieceOfKnowledge conf) throws InvalidParameterException {
		final PoKHelper h = new JenaPoKHelper(conf);
		final List<String> resources = h.getRessOnPredSubj(this.configuration.getServiceUri(), this.configuration.getHasConfiguration());
		if (resources.isEmpty()) {
			throw new InvalidParameterException(Messages.getString(Keys.C_NO_CONFIG, usageContext, this.configuration.getServiceUri(),
					this.configuration.getHasConfiguration()));
		}
		final String configUri = resources.get(0);
		if (new HashSet<String>(resources).size() > 1) {
			this.logger.warn(Messages.getString(Keys.C_MORE_THAN_ONE_CONF, usageContext, this.configuration.getServiceUri(),
					this.configuration.getHasConfiguration(), configUri));
		}
		final List<String> types = h.getRessOnPredSubj(configUri, RDF.TYPE);
		if (!types.contains(this.configuration.getConfigurationType())) {
			this.logger.warn(Messages.getString(Keys.C_NO_TYPE, usageContext, configUri, this.configuration.getConfigurationType()));
		}
		final List<String> linkedContexts = h.getLitsOnPredSubj(configUri, this.configuration.getIsLinkedToContext());
		if (!linkedContexts.contains(usageContext)) {
			this.logger.warn(Messages.getString(Keys.C_NO_UC, usageContext, configUri, this.configuration.getIsLinkedToContext()));
		}
		final List<String> foldersToCrawl = h.getLitsOnPredSubj(configUri, this.configuration.getHasFolderToCrawl());
		foldersToCrawl.addAll(h.getRessOnPredSubj(configUri, this.configuration.getHasFolderToCrawl()));
		if (foldersToCrawl.isEmpty()) {
			throw new InvalidParameterException(Messages.getString(Keys.C_NO_FOLDER, usageContext, configUri, this.configuration.getHasFolderToCrawl()));
		}
		final String folder = foldersToCrawl.get(0);
		if (new HashSet<String>(foldersToCrawl).size() > 1) {
			this.logger.warn(Messages.getString(Keys.C_MORE_THAN_ONE_FOLDER, usageContext, configUri, this.configuration.getHasFolderToCrawl(), folder));
		}
		return folder;
	}



	/**
	 * Checks that the given path denotes an existing readable folder.
	 * 
	 * @param path
	 *            The path to check validity
	 * @param usageContext
	 *            The usageContext, used only for logging purpose.
	 * @return An existing, readable directory.
	 * @throws AccessDeniedException
	 *             If the path does not exist, or is not readable or is not a directory.
	 */
	public static File checkFolder(final String path, final String usageContext) throws AccessDeniedException {
		final File f = new File(path);
		if (!f.exists()) {
			throw new AccessDeniedException(Messages.getString(Keys.C_FOLDER_NOT_EXIST, usageContext, f.getPath()));
		}
		if (!f.isDirectory()) {
			throw new AccessDeniedException(Messages.getString(Keys.C_FILE_NOT_DIRECTORY, usageContext, f.getPath()));
		}
		if (!f.canRead()) {
			throw new AccessDeniedException(Messages.getString(Keys.C_FILE_NOT_READABLE, usageContext, f.getPath()));
		}
		return f;
	}

}
