/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.iterator.messages;

import java.text.MessageFormat;
import java.util.Arrays;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.apache.commons.logging.LogFactory;



/**
 * This class enable to load messages from the resource bundle associated to this class canonical name.
 * It provides to method, one retrieving the message itself and another one using {@link MessageFormat} mechanism for substitution of parameters.
 * 
 * @see MessageFormat
 * @author ymombrun
 * @date 2011-08-17
 */
public final class Messages {


	/**
	 * The resource bundle that contains the messages.
	 */
	private static final ResourceBundle RESOURCE_BUNDLE = ResourceBundle.getBundle(Messages.class.getCanonicalName());


	/**
	 * The error message to be logged in the internationalisation is not working.
	 */
	private final static String I18N_NOT_WORKING = "The internationalisation is not working...";


	/**
	 * The default constructor that should not be used.
	 */
	private Messages() {
		throw new UnsupportedOperationException("The constructor should not been used");
	}


	/**
	 * Lookup in the <code>ResourceBundle</code> for the <code>key</code> in parameter. If <code>key</code> is not found, returns <code>!key!</code>
	 * 
	 * @param key
	 *            The key of the message in the resource file.
	 * @return The value in the resource file (or <code>!key!</code> if not found).
	 */
	public static String getString(final String key) {
		try {
			return Messages.RESOURCE_BUNDLE.getString(key);
		} catch (final MissingResourceException mre) {
			try {
				LogFactory.getLog(Messages.class).warn(Messages.getString(Keys.M_MESSAGE_NOT_FOUND, key), mre);
			} catch (final MissingResourceException mre2) {
				LogFactory.getLog(Messages.class).warn(Messages.I18N_NOT_WORKING);
			}
			return '!' + key + '!';
		}
	}


	/**
	 * Lookup in the <code>ResourceBundle</code> for the <code>key</code> in parameter. The parameters in ellipse are used through the
	 * {@link MessageFormat#format(String, Object...)} method. If <code>key</code> is not found, returns <code>'!'+key'!'+Arrays.toString(params)</code>.
	 * 
	 * @param key
	 *            The key of the message.
	 * @param params
	 *            The array used by the inner message formatter class to replace informations in the message.
	 * @return The value associated to the key in the property file, or <code>'!'+key'!'+Arrays.toString(params)</code> if not found. Using the array in
	 *         parameter to format the message.
	 */
	public static String getString(final String key, final Object... params) {
		final String pattern;
		try {
			pattern = Messages.RESOURCE_BUNDLE.getString(key);
		} catch (final MissingResourceException mre) {
			if (Keys.M_MESSAGE_NOT_FOUND.equals(key)) {
				LogFactory.getLog(Messages.class).warn(Messages.I18N_NOT_WORKING);
			} else {
				try {
					LogFactory.getLog(Messages.class).warn(Messages.getString(Keys.M_MESSAGE_NOT_FOUND, key), mre);
				} catch (final MissingResourceException mre2) {
					LogFactory.getLog(Messages.class).warn(Messages.I18N_NOT_WORKING);
				}
			}
			return '!' + key + '!' + Arrays.toString(params);
		}
		return MessageFormat.format(pattern, params);
	}


}
