/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 CASSIDIAN
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.service.gate;

import gate.CorpusController;
import gate.Factory;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.LogFactory;

/**
 * This class is a singleton to be used to let the configurable and the analyser service communicate.
 * 
 * It holds the mapping between usageContext and gapp file path, and also usageContext and corpuscontroller.
 * 
 * @author ymombrun
 * @date 2010-08-31
 */
public class Configuration {


	/**
	 * The singleton
	 */
	private static Configuration SINGLETON;


	/**
	 * The map of gapp file path by usageContext
	 */
	final private Map<String, String> usContextForGappFilePath = Collections.synchronizedMap(new HashMap<String, String>());


	/**
	 * The map of corpus controller by usageContext
	 */
	final private Map<String, CorpusController> usContextForController = Collections.synchronizedMap(new HashMap<String, CorpusController>());


	/**
	 * The single constructor that shall not be call twice.
	 */
	private Configuration() {
		super();
	}


	/**
	 * @return The singleton instance of Configuration
	 */
	public static synchronized Configuration getInstance() {
		if (Configuration.SINGLETON == null) {
			Configuration.SINGLETON = new Configuration();
		}
		return Configuration.SINGLETON;
	}


	/**
	 * @param usageContext
	 *            The usageContext that shall not be null.
	 * @return The gapp file path
	 */
	public synchronized String getGappFilePath(final String usageContext) {
		return this.usContextForGappFilePath.get(usageContext);
	}


	/**
	 * @param usageContext
	 *            The usageContext tha shall not be null
	 * @param filePath
	 *            The gapp file path
	 */
	public synchronized void setGateApplicationStateFileName(final String usageContext, final String filePath) {
		this.usContextForGappFilePath.put(usageContext, filePath);
	}


	/**
	 * @param usageContext
	 *            The usageContext that shall not be null
	 * @return The controller is one has been set for this usageContext
	 */
	public synchronized CorpusController getController(final String usageContext) {
		return this.usContextForController.get(usageContext);
	}


	/**
	 * @param usageContext
	 *            The usageContext that shall not be null
	 * @param pipeline
	 *            The controller
	 */
	public synchronized void setController(final String usageContext, final CorpusController pipeline) {
		this.usContextForController.put(usageContext, pipeline);
	}


	/**
	 * Remove entries in the two map that where associated with this usageContext.
	 * 
	 * @param usageContext
	 *            The usageContext that shall not be null
	 */
	public synchronized void resetConfiguration(final String usageContext) {
		if (this.usContextForController.containsKey(usageContext)) {
			Factory.deleteResource(this.usContextForController.get(usageContext));
		}
		this.usContextForController.remove(usageContext);
		this.usContextForGappFilePath.remove(usageContext);
		LogFactory.getLog(this.getClass()).info("Configuration for usageContext '" + usageContext + "' removed.");
	}


	/**
	 * Removes every mappings in this object ; moreover it deletes the singleton.
	 */
	public synchronized void resetConfiguration() {
		for (final CorpusController controller : this.usContextForController.values()) {
			Factory.deleteResource(controller);
		}
		this.usContextForController.clear();
		this.usContextForGappFilePath.clear();
		LogFactory.getLog(this.getClass()).info("Every configuration were removed.");
		Configuration.deleteSingleton();
	}


	/**
	 * Removes the singleton inside a static synchronised method to prevent from deadlocks.
	 */
	private static synchronized void deleteSingleton() {
		Configuration.SINGLETON = null;
	}

}
