package org.ow2.weblab.service.translate.moses;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.service.translate.Translator;

/**
 * 
 * MosesParser hide command line usage. Takes a moses.ini file as parameter and
 * enables direct String translation
 * 
 * @author Jérémie Doucy, CASSIDIAN, an EADS company
 * 
 */
public class MosesTranslator implements Translator {

	/**
	 * Escaped regular expression
	 */
	protected String escapedRegExp;

	/**
	 * Default value for escaped regular expression
	 */
	public final static String DEFAULT_ESCAPED_REG_EXP = "\\p{Punct}";

	/**
	 * ISO 639-1 code language
	 */
	protected String langCode;

	/**
	 * Path to the moses.ini file. Paths inside the file must be absolute.
	 */
	protected String mosesIniPath;

	/**
	 * Use it to read the translation response
	 */
	protected BufferedReader br;

	/**
	 * Use it to writes the translation request
	 */
	protected BufferedWriter bw;

	/**
	 * Logger
	 */
	protected static Log LOGGER = LogFactory.getLog(MosesTranslator.class);

	/**
	 * Creates a MosesParser using a moses.ini file.
	 * 
	 * @param mosesIniPath
	 *            path to moses.ini file
	 * @param langCode
	 *            destination language code
	 * @param escapedRegExp
	 *            regular expression used to escape some char not well used by
	 *            moses
	 * @throws IOException
	 *             Something is wrong in the moses configuration file
	 */
	public MosesTranslator(String mosesIniPath, String langCode,
			String escapedRegExp) throws IOException {
		super();
		this.mosesIniPath = mosesIniPath;
		this.langCode = langCode;
		this.escapedRegExp = escapedRegExp;
		LOGGER.trace("Initializing: " + this);
		launchCommand();
	}

	@Override
	public String toString() {
		return mosesIniPath + " for target language " + langCode + " escaping "
				+ escapedRegExp;
	}

	/**
	 * Creates a MosesParser using a moses.ini file with the default escaped
	 * regular expression
	 * 
	 * @param mosesIniPath
	 *            path to moses.ini file
	 * @param langCode
	 *            destination language code
	 * @throws IOException
	 *             Something is wrong in the moses configuration file
	 */
	public MosesTranslator(String mosesIniPath, String langCode)
			throws IOException {
		this(mosesIniPath, langCode, DEFAULT_ESCAPED_REG_EXP);
	}

	/**
	 * Launch the moses -f moses.ini command. Initialize in writer and out
	 * reader.
	 * 
	 * @throws IOException
	 *             something wring when trying to invoke the command line
	 */
	protected void launchCommand() throws IOException {
		Process process = null;
		br = null;
		bw = null;
		List<String> command = new ArrayList<String>();
		command.add("moses");
		command.add("-f");
		command.add(mosesIniPath);
		LOGGER.info("Trying to launch command line: moses -f " + mosesIniPath);
		ProcessBuilder builder = new ProcessBuilder(command);
		process = builder.start();
		InputStream is = process.getInputStream();
		InputStreamReader isr = new InputStreamReader(is);
		br = new BufferedReader(isr);
		bw = new BufferedWriter(new OutputStreamWriter(
				process.getOutputStream()));
		LOGGER.debug("Testing translation (testing): " + translate("testing"));

	}

	/**
	 * Translate into another language using previously defined moses.ini file.
	 * 
	 * @param toBeTranslated
	 *            the String to be translated in
	 * @return the translated String
	 * @throws IOException
	 *             something wrong during the translation
	 */
	public synchronized String translate(String toBeTranslated)
			throws IOException {
		toBeTranslated = cleanString(toBeTranslated);
		LOGGER.debug("Translation of: " + toBeTranslated);
		bw.append(toBeTranslated.toLowerCase());
		bw.newLine();
		bw.flush();
		String result = br.readLine();
		LOGGER.debug("into: " + result + "\nFinished.");
		if (result == null)
			result = toBeTranslated;
		return result.trim();
	}

	protected String cleanString(String toBeCleaned) {
		return toBeCleaned.replaceAll(escapedRegExp, " ");
	}

	/**
	 * Return the translated language code using ISO 639-1
	 * 
	 * @return the translated language
	 */
	public String getLangCode() {
		return langCode;
	}

}
