/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service;

import java.io.File;

import javax.jws.WebService;

import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.jaxb.WebLabMarshaller;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ResourceContainer;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceArgs;
import org.ow2.weblab.core.services.resourcecontainer.LoadResourceReturn;
import org.ow2.weblab.core.services.resourcecontainer.SaveResourceArgs;
import org.ow2.weblab.core.services.resourcecontainer.SaveResourceReturn;


/**
 * This is a simple implementation of a ResourceContainer.
 * 
 * It enable to save resource on a file system and get them after.
 * It is a simpler version that the filerepository one.
 * Main differences are:
 * <ul>
 * <li>This version does not allocate uris; it means that incoming URIs has to be unique otherwise their will be some replacements. It maybe be useful for some
 * applications but in most of the case you'd rather let the FileRepository allocate URIs.</li>
 * <li>This version does not enable to get a subresource. It means that if you just want to change an annotation on a document, should shall re-save the whole
 * document. The FileRepository has this functionality.</li>
 * </ul>
 * 
 * @author ymombrun, Copyright Cassidian
 * @date 2010-04-15
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.ResourceContainer")
public class SimpleRepo implements ResourceContainer {

	/**
	 * The base folder of the repository in the file system
	 */
	protected final File repoBase;

	/**
	 * Instance of WebLabMarshaller used to read resources stored in repo.
	 */
	protected final WebLabMarshaller reader;

	/**
	 * Instance of WebLabMarshaller used to store resources in repo.
	 */
	protected final WebLabMarshaller writer;

	/**
	 * The logger used inside.
	 */
	private final Log logger;


	/**
	 * The constructor.
	 * 
	 * @param path
	 *            The path of the base directory of the repository where resources are stored.
	 */
	public SimpleRepo(final String path) {
		if (path == null) {
			throw new WebLabUncheckedException("Path was null.");
		}
		this.logger = LogFactory.getLog(this.getClass());
		this.repoBase = new File(path);
		if (!this.repoBase.exists() && !this.repoBase.mkdirs()) {
			throw new WebLabUncheckedException("Unable to create repository base directory '" + this.repoBase + "' from parameter '" + path + "'.");
		}
		if (!this.repoBase.isDirectory()) {
			throw new WebLabUncheckedException("A file exists where the base directory should be ('" + this.repoBase + "' from parameter '" + path + "').");
		}
		if (!(this.repoBase.canRead() && this.repoBase.canWrite())) {
			throw new WebLabUncheckedException("The base directory '" + this.repoBase + "' (from parameter '" + path + "') is not readable or not writable.");
		}
		this.reader = new WebLabMarshaller();
		this.writer = new WebLabMarshaller();
		this.logger.info("SimpleRepository successfully started. Using '" + this.repoBase.getAbsolutePath() + "' as base directory.");
	}


	@Override
	public LoadResourceReturn loadResource(final LoadResourceArgs args) throws InvalidParameterException, UnexpectedException {

		// Extract the resource to get
		final String uri = this.checkGetResourceArgs(args);

		this.logger.debug("Try to retrieve resource from URI '" + uri + "'.");

		// Get the abstract file path to read
		final File file = this.uriToFile(uri);

		this.logger.debug("Try to extract resource from file '" + file.getAbsolutePath() + "'.");

		// Extract the resource contained by file
		final Resource res = this.readFile(file, uri);

		this.logger.debug("Return the resource having '" + uri + "' has URI and read from file '" + file.getAbsolutePath() + "'.");

		final LoadResourceReturn grr = new LoadResourceReturn();
		grr.setResource(res);
		return grr;
	}


	@Override
	public SaveResourceReturn saveResource(final SaveResourceArgs args) throws InvalidParameterException, UnexpectedException {
		// Extract Resource to save
		final Resource res = this.checkSaveResourceArgs(args);

		this.logger.debug("Try to save resource with URI '" + res.getUri() + "'.");

		// Get the abstract file path to write
		final File file = this.uriToFile(res.getUri());

		this.logger.debug("File '" + file.getAbsolutePath() + "' will be used.");

		// Write the resource into the file
		this.writeFile(file, res);

		this.logger.debug("Return the URI '" + res.getUri() + "' of the resource that has been successfully stored in file '" + file.getAbsolutePath() + "'.");

		final SaveResourceReturn srr = new SaveResourceReturn();
		srr.setResourceId(res.getUri());
		return srr;
	}

	/**
	 * Try to marshal the resource in file.
	 * 
	 * @param file
	 *            The file that will contain resource
	 * @param res
	 *            The resource to store on file system
	 * @throws If
	 *             res was not marshaled into file (in most of the cast due to an IOException).
	 */
	private synchronized void writeFile(final File file, final Resource res) throws UnexpectedException {
		try {
			this.writer.marshalResource(res, file);
		} catch (final WebLabCheckedException wlce) {
			throw new UnexpectedException("Error when marshalling the resource with URI '" + res.getUri() + "' in file '" + file.getAbsolutePath() + "'.", wlce);
		}
	}

	/**
	 * Checks that args contains a resource with a non null uri.
	 * 
	 * @param args
	 *            The SaveResourceArgs to test
	 * @return The resource contained by args
	 * @throws SaveResourceException
	 *             If args, args.getResource() or args.getResource().getUri() is null.
	 */
	private Resource checkSaveResourceArgs(final SaveResourceArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("SaveResourceArgs is null.");
		}

		if (args.getResource() == null) {
			throw new InvalidParameterException("Resource in SaveResourceArgs is null.");
		}
		final String uri = args.getResource().getUri();
		if (uri == null) {
			final InvalidParameterException wle = new InvalidParameterException("URI of the Resource in SaveResourceArgs is null.");
			throw wle;
		}

		return args.getResource();
	}


	/**
	 * Try to unmarshal a resource from file.
	 * 
	 * @param file
	 *            The file to be unmarshaled into a resource
	 * @param uri
	 *            The uri to be used for logs
	 * @return The resource contained in file
	 * @throws GetResourceException
	 *             If was not unmarshaled (in most of the cast due to an IOException).
	 */
	private synchronized Resource readFile(final File file, final String uri) throws UnexpectedException {
		if (!file.exists()) {
			throw new UnexpectedException("No resource with '" + uri + "' as URI found. File " + file.getAbsolutePath() + " not found.");
		}
		try {
			return this.reader.unmarshal(file, Resource.class);
		} catch (final WebLabCheckedException wlce) {
			throw new UnexpectedException("Error when unmarshalling the file" + file.getAbsolutePath() + " from URI '" + uri + "'.", wlce);
		}
	}


	/**
	 * Hash the URI to obtain a path to an unique file to be used to store (or load) the resource.
	 * 
	 * @param uri
	 *            The URI of the Resource to get the storing file
	 * @return The file used to store a resource
	 */
	protected File uriToFile(final String uri) {
		// Creates a String of 64 chars
		final String sha = DigestUtils.sha256Hex(uri);

		// To balance the files, in more than one directory, creates a directory having the first SHA char has name.
		final File folder = new File(this.repoBase, sha.substring(0, 1));
		if (!folder.exists() && !folder.mkdirs()) {
			this.logger.warn("An error occurs while creating folder " + folder.getAbsolutePath() + " used to store resource with " + uri + " as URI.");
		}

		// The file in folder.
		return new File(folder, sha + ".xml");
	}

	/**
	 * Checks that the args is correct. I.e. contains a non null URI.
	 * 
	 * @param args
	 *            Wrapper of resourceId
	 * @return The resourceId in args
	 * @throws GetResourceException
	 *             If args or resourceId in args is null.
	 */
	private String checkGetResourceArgs(final LoadResourceArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("LoadResourceArgs is null");
		}
		final String uri = args.getResourceId();
		if (uri == null) {
			throw new InvalidParameterException("uri inLoadResourceArgs is null");
		}
		return uri;
	}


	
}
