/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service.gazetteer.api;

import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;

import org.apache.commons.logging.Log;
import org.ow2.weblab.core.extended.factory.AnnotationFactory;
import org.ow2.weblab.core.extended.factory.SegmentFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabProcessing;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;

public abstract class TextAnnotator implements Analyser {


	protected Log logger;


	public TextAnnotator(final Log logger) {
		super();
		this.logger = logger;
	}


	@Override
	public ProcessReturn process(final ProcessArgs args) throws AccessDeniedException, ContentNotAvailableException, InsufficientResourcesException,
			InvalidParameterException, ServiceNotConfiguredException, UnexpectedException, UnsupportedRequestException {

		final List<Text> texts = this.checkParameters(args);
		this.logger.debug("Start processing resource " + args.getResource() + ". It contains " + texts.size() + " texts.");

		for (final Text text : texts) {
			this.process(text);
		}

		final ProcessReturn ret = new ProcessReturn();
		ret.setResource(args.getResource());
		return ret;
	}


	/**
	 * Create an annotation on a text based on element detected.
	 * 
	 * @param text
	 *            : the Text unit being annotated
	 * @param offset
	 *            : the offset of the detected element
	 * @param word
	 *            : the word being detected
	 * @param className
	 *            : the class name of the detected object
	 * @param label
	 *            : the canonical label to use
	 * @param createAnnotation
	 *            : boolean option to force creation of Annotation for each new call or reuse the first Annotation on the Text.
	 */
	public void annote(final Text text, final int offset, final String word, final String className, final String label, final boolean createAnnotation) {
		this.logger.debug("Creating segment : " + offset + " - " + word);
		final LinearSegment linearSegment = SegmentFactory.createAndLinkLinearSegment(text, offset, offset + word.length());

		Annotation annotation = null;
		if (!createAnnotation) {
			if (text.getAnnotation().size() > 0) {
				// TODO in the case we decide to not create a new Annotation, we update the first one... Is it ok ?
				annotation = text.getAnnotation().get(0);
			}
		}
		if (annotation == null) {
			annotation = AnnotationFactory.createAndLinkAnnotation(text);
		}

		final String instanceUri = WebLabProcessing.NAMESPACE + "temp-" + new Date().getTime();

		final PoKHelper h = new JenaPoKHelper(annotation);
		h.setAutoCommitMode(false);
		h.createResStat(linearSegment.getUri(), WebLabProcessing.REFERS_TO, instanceUri);
		h.createResStat(instanceUri, RDF.TYPE, WebLabProcessing.NAMESPACE + "demo-" + className);
		h.createLitStat(instanceUri, RDFS.LABEL, label);
		h.createLitStat(instanceUri, WebLabProcessing.IS_CANDIDATE, "true");

		h.commit();

		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Word annotated : " + word + " class " + className + " -> " + offset);
			this.logger.debug("In text : '" + text.getContent().substring(offset, offset + word.length()) + "'");
		}

	}


	public abstract void process(Text text) throws AccessDeniedException, ContentNotAvailableException, InsufficientResourcesException,
			InvalidParameterException, ServiceNotConfiguredException, UnexpectedException, UnsupportedRequestException;


	/**
	 * @param args
	 *            The ProcessArgs
	 * @return The list of Text contained by the Resource in args.
	 * @throws ProcessException
	 *             For any reason preventing the retrieval of text unit to be done.
	 */
	protected List<Text> checkParameters(final ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("ProcessArgs was null.", "ProcessArgs was null.");
		}

		final Resource res = args.getResource();
		if (res == null) {
			throw new InvalidParameterException("Resource in ProcessArg was null.", "Resource in ProcessArg was null.");
		}
		if (!(res instanceof MediaUnit)) {
			throw new InvalidParameterException("This service only process MediaUnit; Resource was a: " + res.getClass().getSimpleName() + ".",
					"This service only process MediaUnit; Resource was a: " + res.getClass().getSimpleName() + ".");
		}

		final List<Text> texts;
		if (res instanceof Text) {
			texts = new LinkedList<Text>();
			texts.add((Text) res);
		} else {
			texts = ResourceUtil.getSelectedSubResources(args.getResource(), Text.class);
		}

		// Removing empty text to prevent useless processing
		for (final ListIterator<Text> textIt = texts.listIterator(); textIt.hasNext();) {
			final Text text = textIt.next();
			if ((text.getContent() == null) || text.getContent().replaceAll("\\s+", "").isEmpty()) {
				textIt.remove();
			}
		}

		return texts;
	}


}
