/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service.gazetteer.api;

import java.util.Date;

import org.apache.commons.logging.Log;
import org.ow2.weblab.core.extended.factory.AnnotationFactory;
import org.ow2.weblab.core.extended.factory.SegmentFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLab;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.RDFHelperFactory;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;

public abstract class TextAnnotator implements Analyser {

	protected Log logger;

	public TextAnnotator(Log logger) {
		super();
		this.logger = logger;
	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, ContentNotAvailableException, InsufficientResourcesException, InvalidParameterException, ServiceNotConfiguredException, UnexpectedException, UnsupportedRequestException {
		Resource resource = args.getResource();

		for (Text text : ResourceUtil.getSelectedSubResources(resource, Text.class)) {
			this.logger.info("Processing text.");
			this.process(text);
		}

		ProcessReturn ret = new ProcessReturn();
		ret.setResource(resource);
		return ret;
	}

	/**
	 * Create an annotation on a text based on element detected. 
	 * @param text : the Text unit being annotated
	 * @param offset : the offset of the detected element
	 * @param word : the word being detected
	 * @param className : the class name of the detected object
	 * @param label : the canonical label to use
	 * @param createAnnotation : boolean option to force creation of Annotation for each new call or reuse the first Annotation on the Text.
	 */
	public void annote(final Text text, final int offset, final String word, final String className, final String label, boolean createAnnotation) {
		this.logger.debug("Creating segment : " + offset + " - " + word);
		LinearSegment linearSegment = SegmentFactory.createAndLinkLinearSegment(text, offset, offset + word.length());
		
		Annotation annotation = null;
		if(!createAnnotation){
			if(text.getAnnotation().size() > 0){
				//TODO in the case we decide to not create a new Annotation, we update the first one... Is it ok ?
				annotation = text.getAnnotation().get(0);
			}
		}
		if(annotation == null)
			annotation = AnnotationFactory.createAndLinkAnnotation(text);
			
		final String instanceUri = WebLab.PROCESSING_PROPERTY_NAMESPACE + "temp-" + new Date().getTime();

		PoKHelper h = RDFHelperFactory.getPoKHelper(annotation);
		h.setAutoCommitMode(false);
		h.createResStat(linearSegment.getUri(), WebLab.REFERS_TO, instanceUri);
		h.createResStat(instanceUri, RDF.TYPE, WebLab.PROCESSING_PROPERTY_NAMESPACE + "demo-" + className);
		h.createLitStat(instanceUri, RDFS.LABEL, label);
		h.createLitStat(instanceUri, WebLab.IS_CANDIDATE, "true");

		h.commit();

		if (this.logger.isDebugEnabled()) {
			this.logger.debug("Word annotated : " + word + " class " + className + " -> " + offset);
			this.logger.debug("In text : '" + text.getContent().substring(offset, offset + word.length()) + "'");
		}

	}

	public abstract void process(Text text) throws AccessDeniedException, ContentNotAvailableException, InsufficientResourcesException, InvalidParameterException, ServiceNotConfiguredException, UnexpectedException, UnsupportedRequestException;

}
