/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2012 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.dataoperator.impl;

import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.DataOperator;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.dataoperator.DeleteArgs;
import org.ow2.weblab.core.services.dataoperator.DeleteReturn;
import org.ow2.weblab.core.services.dataoperator.DropArgs;
import org.ow2.weblab.core.services.dataoperator.DropReturn;
import org.ow2.weblab.core.services.dataoperator.ReportArgs;
import org.ow2.weblab.core.services.dataoperator.ReportReturn;
import org.ow2.weblab.services.indexer.impl.SolrIndexer;

/**
 * Has the same config than the indexer 
 * 
 * @author asaval
 *
 */
@WebService(endpointInterface = "org.ow2.weblab.core.services.DataOperator")
public class SolrDataOperator extends SolrIndexer implements DataOperator {


	private final static Log logger = LogFactory.getLog(SolrDataOperator.class);
	
	@Override
	public DeleteReturn delete(DeleteArgs args)
			throws ServiceNotConfiguredException,
			InsufficientResourcesException, UnexpectedException,
			InvalidParameterException, AccessDeniedException,
			ContentNotAvailableException, UnsupportedRequestException {
		
		String uri = args.getUri();
		if (uri == null){
			logger.error("Null URI send to the Solr dataoperator");
			throw new InvalidParameterException("Uri can not be null.");
		}
		try{
			this.solr.open(this.getWebAppPath());

			// add quotes to avoid uri interpretation by solr
			this.solr.removeDocument("\""+uri+"\"");
			
		}catch(WebLabCheckedException e){
			logger.error(e,e);
			throw new UnexpectedException(e.getMessage(),e);
		}
		
		return new DeleteReturn();
	}

	@Override
	public DropReturn drop(DropArgs args) throws ServiceNotConfiguredException,
			InsufficientResourcesException, UnexpectedException,
			InvalidParameterException, AccessDeniedException,
			ContentNotAvailableException, UnsupportedRequestException {
		try{
			this.solr.open(this.getWebAppPath());

			this.solr.removeDocument("weblab*");
			
		}catch(WebLabCheckedException e){
			logger.error(e,e);
			throw new UnexpectedException(e.getMessage(),e);
		}
		
		return new DropReturn();
	}

	@Override
	public ReportReturn report(ReportArgs args)
			throws ServiceNotConfiguredException,
			InsufficientResourcesException, UnexpectedException,
			InvalidParameterException, AccessDeniedException,
			ContentNotAvailableException, UnsupportedRequestException {
		throw new UnsupportedRequestException("This operation is not implemented");
	}
	
	@PreDestroy
	public void destroy() {
		if (this.solr != null) {
			this.solr.close();
		}
	}

}
