/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.searcher.impl;

import java.io.File;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.annotation.PreDestroy;
import javax.jws.WebService;
import javax.mail.search.SearchException;
import javax.servlet.ServletContext;
import javax.xml.ws.WebServiceContext;
import javax.xml.ws.handler.MessageContext;

import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabModel;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.BeanHelper;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.Searcher;
import org.ow2.weblab.core.services.searcher.SearchArgs;
import org.ow2.weblab.core.services.searcher.SearchReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.util.index.Field;
import org.ow2.weblab.util.search.SearcherConfig;

/**
 * Searcher using Embedded Solr server.<br/>
 * RDF output is generated with Jena and can be enriched with metadata stored in
 * index. This should be define in searcher configuration file.
 */

@WebService(endpointInterface = "org.ow2.weblab.core.services.Searcher")
public class SolrSearcher implements Searcher {


	@javax.annotation.Resource
	protected WebServiceContext wsContext;


	private static int resultsCpt = 0;


	private final SolrSearcherConfig conf;


	private final SolrComponent solrComponent;


	private final static Log logger = LogFactory.getLog(SolrSearcher.class);


	public SolrSearcher() {
		this.conf = BeanHelper.getInstance().getSpecificInstance(SolrSearcherConfig.DEFAULT_CONF_FILE, true)
				.getBean(SearcherConfig.DEFAULT_BEAN_NAME, SolrSearcherConfig.class);
		this.solrComponent = SolrComponent.getInstance();
	}


	@PreDestroy
	public void destroy() {
		if (this.solrComponent != null) {
			this.solrComponent.close();
		}
	}


	@Override
	public SearchReturn search(final SearchArgs arg0) throws InvalidParameterException {
		final Query q = arg0.getQuery();

		final int offset;
		if ((arg0.getOffset() != null) && (arg0.getOffset().intValue() > 0)) {
			offset = arg0.getOffset().intValue();
		} else {
			offset = 0;
		}

		final int limit;
		if (((arg0.getLimit() != null) && (arg0.getLimit().intValue() > 0))) {
			limit = arg0.getLimit().intValue();
		} else {
			limit = 10;
		}

		final SearchReturn re = new SearchReturn();
		re.setResultSet(this.search(q, offset, limit));
		return re;
	}


	/**
	 * Use <code>SolrComponent</code> for Solr index querying and format
	 * response in RDF with Jena.<br/>
	 * The ResultSet contains a Pok which himself contains Hits.<br/>
	 * If enrichment is activated, all fields presents in index and search
	 * configuration file are annotated.
	 * 
	 * @param q
	 *            the Weblab <code>StringQuery</code>
	 * @param offset
	 *            results start index
	 * @param limit
	 *            results end index
	 * @return the <code>ResultSet</code> containing hits
	 * @throws SearchException
	 */
	public ResultSet search(final Query q, final int offset, final int limit) throws InvalidParameterException {
		StringQuery query = null;

		if (q instanceof StringQuery) {
			query = (StringQuery) q;
		} else {
			throw new InvalidParameterException("This service " + this.getClass().getSimpleName() + " can only process " + StringQuery.class.getSimpleName()
					+ ".", ("Input has invalid type:" + q.getClass()) == null ? " no xsi:type !" : q.getClass().getSimpleName());
		}

		try {
			SolrSearcher.logger.debug("query request : " + query.getRequest());
			SolrSearcher.logger.debug("Query input : " + ResourceUtil.saveToXMLString(query));

		} catch (final WebLabCheckedException e) {
			SolrSearcher.logger.debug("Error when logging query input", e);
		}

		final SearchReturn returnedResults = new SearchReturn();

		final ResultSet results = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF,
				SolrSearcherConfig.DEFAULT_IDRES + SolrSearcher.resultsCpt++, ResultSet.class);

		final PieceOfKnowledge pok = ResourceFactory.createResource(SolrSearcherConfig.DEFAULT_IDREF, SearcherConfig.DEFAULT_IDPOK + SolrSearcher.resultsCpt++,
				PieceOfKnowledge.class);

		results.setPok(pok);

		final PoKHelper h = new JenaPoKHelper(pok);
		h.setAutoCommitMode(false);
		returnedResults.setResultSet(results);

		/*
		 * Set namespace prefixes
		 */
		for (final String nsPrefix : this.conf.getNsPrefixProperties().keySet()) {
			h.setNSPrefix(nsPrefix, this.conf.getNsPrefixProperties().get(nsPrefix));
		}

		/*
		 * Set Result type property
		 */
		h.createResStat(results.getUri(), RDF.TYPE, WebLabModel.RESULT_SET);

		/*
		 * Set ResultSet "isResultOf" property
		 */
		h.createResStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.RESULTS_FROM_KEY), q.getUri());

		/*
		 * Set ResultSet "rdf:label" property
		 */
		for (final Entry<String, String> local : this.conf.getResultSetLabels().entrySet()) {
			h.createLitStat(results.getUri(), RDFS.LABEL, local.getValue(), local.getKey());
		}

		try {
			/*
			 * Open Solr searcher
			 */
			this.solrComponent.open(this.getWebAppPath());

			final QueryResponse res = this.solrComponent.search(query.getRequest(), offset, limit);

			int resCpt = 0;

			// create a Document for each result
			final SolrDocumentList resultDocs = res.getResults();
			final Map<String, Map<String, List<String>>> highlightings = res.getHighlighting();

			if (resultDocs != null) {
				for (final SolrDocument hit : resultDocs) {
					final String resourceUri = String.valueOf(hit.getFieldValue("id"));
					final String hitUri = "weblab://" + SolrSearcherConfig.DEFAULT_IDREF + "/" + SearcherConfig.DEFAULT_IDHIT + resCpt;

					h.createResStat(results.getUri(), WebLabRetrieval.HAS_HIT, hitUri);
					h.createResStat(hitUri, RDF.TYPE, this.conf.getHitClass());

					// adding hasRank annotation
					h.createLitStat(hitUri, this.conf.getProperties().get(SearcherConfig.HAS_RANK_KEY), String.valueOf(offset + resCpt + 1));

					// Solr score
					h.createLitStat(hitUri, this.conf.getProperties().get("hasScore"), String.valueOf(hit.getFieldValue("score")));

					// Generate snippet from hit highlightings
					if (highlightings != null) {
						if (highlightings.get(hit.getFieldValue("id")).size() > 0) {
							final List<String> hitHighlightingList = highlightings.get(hit.getFieldValue("id")).get("text");
							final StringBuffer snippet = new StringBuffer();
							for (final String hl : hitHighlightingList) {
								snippet.append(hl);
							}
							h.createLitStat(hitUri, this.conf.getProperties().get(SearcherConfig.DESCRIPTION_KEY), snippet.toString());
						}
					}

					h.createResStat(hitUri, this.conf.getProperties().get(SearcherConfig.LINKED_TO_KEY), resourceUri);
					h.createResStat(hitUri, this.conf.getProperties().get(SearcherConfig.IN_RESULT_SET_KEY), results.getUri());

					// Meta enrichment
					if (this.conf.isHitsEnrichedWithMetas()) {
						final Map<String, Collection<Object>> fields = hit.getFieldValuesMap();

						for (final Entry<String, Field> confField : this.conf.getFieldsToRetrieve().entrySet()) {
							final Field field = confField.getValue();

							if (fields.containsKey(field.getName())) {
								for (final String property : field.getProperties()) {
									final Collection<Object> fieldValues = fields.get(field.getName());
									for (final Object fieldValue : fieldValues) {
										if (fieldValue instanceof String) {
											if (field.getType().equals("URI")) {
												h.createResStat(resourceUri, property, ((String) fieldValue).trim());
											} else {
												h.createLitStat(resourceUri, property, ((String) fieldValue).trim());
											}
										} else if (fieldValue instanceof Date) {
											h.createLitStat(resourceUri, property, DateFormatUtils.ISO_DATETIME_TIME_ZONE_FORMAT.format((Date) fieldValue));
										}
									}
								}
							}
						}
					}

					resCpt++;
				}
				h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.NB_RESULTS_KEY), String.valueOf(resultDocs.getNumFound()));
			} else {
				h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.NB_RESULTS_KEY), "0");
			}

			h.createLitStat(results.getUri(), this.conf.getProperties().get(SearcherConfig.OFFSET_KEY), String.valueOf(offset));

			/*
			 * BEGIN FACET CODE !!!!
			 */
			if (this.conf.getFacetFields().size() > 0) {
				int facetCpt = 0;
				for (final String facetFieldName : this.conf.getFacetFields()) {
					final String facetUri = "weblab://" + SolrSearcherConfig.DEFAULT_IDREF + SearcherConfig.DEFAULT_IDFACET + SolrSearcher.resultsCpt + "_"
							+ facetCpt++;

					h.createResStat(facetUri, RDF.TYPE, this.conf.getFacetClass());
					h.createLitStat(facetUri, RDFS.LABEL, facetFieldName);
					h.createResStat(facetUri, this.conf.getIsLinkedToQueryClass(), q.getUri());

					final FacetField ff = res.getFacetField(facetFieldName);
					if (ff != null) {
						final List<Count> countList = ff.getValues();
						if (countList != null) {
							int facetValueCpt = 0;
							for (final Count count : countList) {
								SolrSearcher.logger.debug("Create facet element: " + facetFieldName + "=" + count.getName() + "(" + count.getCount() + ")");

								final String facetValueUri = "weblab://" + SolrSearcherConfig.DEFAULT_IDREF + SearcherConfig.DEFAULT_IDFACETVALUE
										+ SolrSearcher.resultsCpt + "_" + facetCpt + "_" + facetValueCpt++;

								h.createResStat(facetValueUri, RDF.TYPE, this.conf.getFacetValueClass());
								h.createLitStat(facetValueUri, RDFS.LABEL, count.getName());
								h.createLitStat(facetValueUri, this.conf.getFacetCountClass(), String.valueOf(count.getCount()));
								h.createLitStat(facetValueUri, this.conf.getFacetFilterQueryClass(), count.getAsFilterQuery());
								h.createResStat(facetValueUri, this.conf.getIsLinkedToFacetClass(), facetUri);
							}
						}
					}
				}
			}

			/* END FACET CODE */
			h.commit();
		} catch (final WebLabCheckedException wlce) {
			throw new InvalidParameterException("Cannot retrieve the results to query [" + query.getRequest() + "] - " + wlce.getMessage(), wlce);
		}

		return results;
	}


	/**
	 * Return the Web application path
	 * 
	 * @return webapp path
	 * @throws WebLabCheckedException
	 */
	protected String getWebAppPath() throws WebLabCheckedException {

		String appPath = null;
		if (this.wsContext != null) {
			final ServletContext ctx = (ServletContext) this.wsContext.getMessageContext().get(MessageContext.SERVLET_CONTEXT);
			appPath = ctx.getRealPath("WEB-INF/classes");
		} else {
			SolrSearcher.logger.warn("Webservice context not available returning current local path as default web app path");
		}

		if (appPath == null) {
			throw new WebLabCheckedException("Webapp path [" + appPath + "] is null...");
		}
		if (!new File(appPath).exists()) {
			throw new WebLabCheckedException("Webapp path [" + appPath + "] does not exists...");
		}

		return appPath;
	}


	/**
	 * @return the wsContext
	 */
	public WebServiceContext getWsContext() {
		return this.wsContext;
	}


	/**
	 * @param wsContext
	 *            the wsContext to set
	 */
	public void setWsContext(final WebServiceContext wsContext) {
		this.wsContext = wsContext;
	}

}
