/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.solr.searcher;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;
import javax.mail.search.SearchException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.apache.solr.common.SolrDocumentList;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.factory.PoKFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.WebLabModel;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.helper.impl.JenaResourceHelper;
import org.ow2.weblab.core.model.PieceOfKnowledge;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.SimilarityQuery;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.model.retrieval.WRetrievalAnnotator;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.Searcher;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.searcher.SearchArgs;
import org.ow2.weblab.core.services.searcher.SearchReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.analyser.FacetSuggestion;
import org.ow2.weblab.services.solr.analyser.Highlighter;
import org.ow2.weblab.services.solr.analyser.ResultSetMetadataEnrichment;
import org.ow2.weblab.services.solr.indexer.SolrIndexerConfig;

/**
 * Searcher using Embedded Solr server.<br/>
 * RDF output is generated with Jena and can be enriched with metadata stored in
 * index. This should be define in searcher configuration file.
 */

@WebService(endpointInterface = "org.ow2.weblab.core.services.Searcher")
public class SolrSearcher implements Searcher {
	// TODO these two properties will be part of model 1.2.5. Use annotator when
	// this is ready.
	public static final String TO_BE_ORDERED_BY = "http://weblab.ow2.org/core/1.2/ontology/retrieval#toBeOrderedBy";
	public static final String ASCENDENT_ORDERING_MODE_EXPECTED = "http://weblab.ow2.org/core/1.2/ontology/retrieval#ascendentOrderingModeExpected";

	public static final String BEAN_NAME = "searcherServiceBean";
	private static int resultsCounter = 0;
	private static int hitCounter = 0;

	private Log logger = LogFactory.getLog(SolrSearcher.class);
	private SolrIndexerConfig indexerConfig;
	private String solrURL;

	// these two fields are only here to please Yann ;-)
	private ResultSetMetadataEnrichment enricher;
	private Highlighter highlighter;
	private FacetSuggestion facetSuggestion;

	private boolean noCore = false;

	@PostConstruct
	public void init() {
		// init default instance
		SolrComponent.getInstance(solrURL, null);

		// get indexingConfig if not loaded
		if (indexerConfig == null) {
			// get it directly ?
		}

		try {
			new URL(solrURL);
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + solrURL + "].", e);
		}
	}

	@PreDestroy
	public void destroy() {
		// nothing to do I guess
	}

	@Override
	public SearchReturn search(final SearchArgs arg) throws InvalidParameterException, ServiceNotConfiguredException, UnexpectedException {
		Query q = arg.getQuery();

		final Integer offset;
		if (arg.getOffset() != null && arg.getOffset() > 0)
			offset = arg.getOffset();
		else
			offset = new Integer(0);

		final Integer limit;
		if ((arg.getLimit() != null && arg.getLimit() > 0))
			limit = arg.getLimit();
		else
			limit = 10;

		ResultSet set = this.search(arg.getUsageContext(), q, offset, limit);

		// retro-compatibility code for old all-in-solr model of retrieval
		if (enricher != null) {
			set = enricher.addMetadataToResultSet(arg.getUsageContext(), set);
		}

		if (highlighter != null) {
			set = highlighter.highLightHitInResultSet(arg.getUsageContext(), set);
		}

		if (facetSuggestion != null) {
			set = facetSuggestion.doFacetSuggest(arg.getUsageContext(), set);
		}
		// end of retro-compatibility code

		SearchReturn re = new SearchReturn();
		re.setResultSet(set);
		return re;
	}

	/**
	 * Use <code>SolrComponent</code> for Solr index querying and format
	 * response in RDF with Jena.<br/>
	 * The ResultSet contains a Pok which himself contains Hits.<br/>
	 * If enrichment is activated, all fields presents in index and search
	 * configuration file are annotated.
	 * 
	 * @param q
	 *            the Weblab <code>StringQuery</code>
	 * @param offset
	 *            results start index
	 * @param limit
	 *            results end index
	 * @return the <code>ResultSet</code> containing hits
	 * @throws ServiceNotConfiguredException
	 * @throws UnexpectedException
	 * @throws SearchException
	 */
	public ResultSet search(final String context, final Query q, final int offset, final int limit) throws InvalidParameterException,
			ServiceNotConfiguredException, UnexpectedException {
		if (!(q instanceof StringQuery) && !(q instanceof SimilarityQuery)) {
			throw new InvalidParameterException("This service " + this.getClass().getSimpleName() + " can only process " + StringQuery.class.getSimpleName()
					+ ".", "Input has invalid type:" + q.getClass() == null ? " no xsi:type !" : q.getClass().getSimpleName());
		}

		try {
			if (q instanceof StringQuery) {
				logger.debug("String query request : " + ((StringQuery) q).getRequest());
			} else if (q instanceof SimilarityQuery) {
				logger.debug("Sim query request : " + ((SimilarityQuery) q).getResource().size());
			}
			if (logger.isTraceEnabled()) {
				logger.trace("Query input : " + ResourceUtil.saveToXMLString(q));
			}
		} catch (WebLabCheckedException e) {
			logger.debug("Error when logging query input", e);
		}

		try {
			SolrComponent instance;
			if (noCore) {
				instance = SolrComponent.getInstance(solrURL, null);
			} else {
				instance = SolrComponent.getInstance(solrURL, context);
			}
			QueryResponse res = null;

			// check if the query has annotation to expect a specific behavior
			boolean ascOrder = false;
			String orderedByProperty = null;
			if (q.getAnnotation().size() > 0) {
				ascOrder = getOrder(q);
				orderedByProperty = getOrderBy(q);
			}

			if (q instanceof StringQuery) {
				// querying solr engine with simple text query
				res = instance.search(((StringQuery) q).getRequest(), offset, limit, orderedByProperty, ascOrder ? ORDER.asc : ORDER.desc);
			} else if (q instanceof SimilarityQuery) {
				// launching the similarity query for solr
				res = launchSimilarityQuery(q, offset, limit, instance);
			}

			final SolrDocumentList resultDocs = res.getResults();
			int resCpt = 0;

			ResultSet results = ResourceFactory.createResource(SolrComponent.IDREF, SolrComponent.IDRES_RESULT_PREFIX + resultsCounter++, ResultSet.class);
			// adding the original query as first resource in the resultSet
			results.getResource().add(q);
			URI queryURI = new URI(q.getUri());

			// adding pok
			PieceOfKnowledge pok = PoKFactory.createAndLinkPoK(results);
			results.setPok(pok);
			// starting to write RDF information in pok
			WRetrievalAnnotator wra = new WRetrievalAnnotator(new URI(results.getUri()), pok);

			// Set Result type property
			wra.writeType(new URI(WebLabModel.RESULT_SET));
			// Set ResultSet "isResultOf" property
			wra.writeResultOf(queryURI);
			// writing expected offset/limit
			wra.writeExpectedOffset(offset);
			wra.writeExpectedLimit(limit);

			// check if there are some results
			if (resultDocs == null || resultDocs.getNumFound() == 0) {
				// adding no results information
				wra.writeNumberOfResults(0);
			} else {
				// adding total number of results annotation
				wra.writeNumberOfResults((int) resultDocs.getNumFound());
				// create Hit annotations for each result
				for (final SolrDocument hit : resultDocs) {
					final URI resourceUri = new URI(String.valueOf(hit.getFieldValue("id")));
					final URI hitUri = new URI("weblab://" + SolrComponent.IDREF + "/" + SolrComponent.IDRES_HIT_PREFIX + hitCounter++);
					// linking resultSet to the Hit
					wra.writeHit(hitUri);
					// starting Hit specific annotation
					wra.startInnerAnnotatorOn(hitUri);
					// adding type annotation
					wra.writeType(new URI(WebLabRetrieval.HIT));
					// adding hasRank annotation
					wra.writeRank(offset + resCpt + 1);
					// adding solr score annotation
					wra.writeScore(0.0 + (Float) hit.getFieldValue("score"));
					// adding link to resource
					wra.writeLinkedTo(resourceUri);

					wra.endInnerAnnotator();
					resCpt++;
				}
			}

			if (logger.isTraceEnabled()) {
				logger.trace("Results input : " + ResourceUtil.saveToXMLString(results));
			}

			return results;
		} catch (WebLabCheckedException e) {
			throw new InvalidParameterException("Cannot retrieve the results to query [" + q.getUri() + "] - " + e.getMessage(), e);
		} catch (URISyntaxException e) {
			throw new UnexpectedException("Invalid URI encountered during RDF annotation. That's really strange isn't it ?", e);
		}
		// unreachable part.
	}

	protected QueryResponse launchSimilarityQuery(final Query q, final int offset, final int limit, SolrComponent instance) throws WebLabCheckedException {
		QueryResponse res;
		StringBuffer queryString = new StringBuffer();
		List<Resource> resources = ((SimilarityQuery) q).getResource();

		if (resources.size() == 1) {
			// single sample => can use more like this function
			queryString.append(SolrIndexerConfig.ID_FIELD);
			queryString.append(':');
			queryString.append('"');
			queryString.append(resources.get(0).getUri());
			queryString.append('"');
			res = instance.moreLikeThis(queryString.toString(), offset, limit);
		} else {
			// multiple samples, thus try simple ID based query
			// TODO check this is the expected behaviour
			for (int i = 0; i < resources.size(); i++) {
				Resource r = resources.get(i);
				if (i > 0) {
					queryString.append(" OR ");
				}
				queryString.append(SolrIndexerConfig.ID_FIELD);
				queryString.append(':');
				queryString.append('"');
				queryString.append(r.getUri());
				queryString.append('"');
			}
			res = instance.search(queryString.toString(), offset, limit);
		}
		return res;
	}

	private String getOrderBy(Query q) {
		ResourceHelper hlpr = new JenaResourceHelper(q);
		List<String> values = hlpr.getRessOnPredSubj(q.getUri(), TO_BE_ORDERED_BY);
		if (values.size() != 1) {
			logger.info("Query is holding multiple values for [" + TO_BE_ORDERED_BY + "]. That's not accepted, so we ignore all of them.");
		} else {
			if (indexerConfig != null){
				return indexerConfig.getPropertyToFieldMap().get(values.get(0));
			}
		}
		return null;
	}

	private boolean getOrder(Query q) {
		ResourceHelper hlpr = new JenaResourceHelper(q);
		List<String> values = hlpr.getLitsOnPredSubj(q.getUri(), ASCENDENT_ORDERING_MODE_EXPECTED);
		if (values.size() != 1) {
			logger.info("Query is holding multiple values for [" + ASCENDENT_ORDERING_MODE_EXPECTED + "]. That's not accepted, so we ignore all of them.");
		} else {
			return Boolean.parseBoolean(values.get(0));
		}
		return false;
	}

	public SolrIndexerConfig getIndexerConfig() {
		return indexerConfig;
	}

	public void setIndexerConfig(SolrIndexerConfig indexerConfig) {
		this.indexerConfig = indexerConfig;
	}

	public String getSolrURL() {
		return solrURL;
	}

	public void setSolrURL(String solrURL) {
		this.solrURL = solrURL;
	}

	public ResultSetMetadataEnrichment getEnricher() {
		return enricher;
	}

	public void setEnricher(ResultSetMetadataEnrichment enricher) {
		this.enricher = enricher;
	}

	public Highlighter getHighlighter() {
		return highlighter;
	}

	public void setHighlighter(Highlighter highlighter) {
		this.highlighter = highlighter;
	}

	public FacetSuggestion getFacetSuggestion() {
		return facetSuggestion;
	}

	public void setFacetSuggestion(FacetSuggestion facetSuggestion) {
		this.facetSuggestion = facetSuggestion;
	}

	public boolean isNoCore() {
		return noCore;
	}

	public void setNoCore(boolean noCore) {
		this.noCore = noCore;
	}

}
