/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2009 EADS DEFENCE AND SECURITY SYSTEMS
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */

package org.ow2.weblab.services.solr;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CommonsHttpSolrServer;
import org.apache.solr.client.solrj.request.CoreAdminRequest;
import org.apache.solr.client.solrj.request.CoreAdminRequest.Create;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.params.CommonParams;
import org.apache.solr.common.params.CoreAdminParams.CoreAdminAction;
import org.apache.solr.common.params.MoreLikeThisParams;
import org.apache.solr.common.util.NamedList;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.MediaUnit;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.services.solr.indexer.SolrIndexerConfig;

/**
 * Component used to :
 * <ul>
 * <li>open a connection to remote solr server</li>
 * <li>add a document to the index</li>
 * <li>search in the index by text and structured queries</li>
 * <li>search documents similar to an existing document (more like this)</li>
 * <li>delete one document from the index</li>
 * <li>drop a complete index</li>
 * </ul>
 * 
 * This class implements singleton pattern in order to share index connection
 * and improve performance (Solr server is thread-safe).<br/>
 * The index buffer is managed by Solr so we just use a counter to commit
 * documents when buffer configured size is achieved.
 */

public class SolrComponent {
	// //////////////////////////////////////////////////////////////////
	// static part
	// //////////////////////////////////////////////////////////////////

	/**
	 * Default query handler defined in solrconfig.xml This one should only
	 * enable to get list of results URI and scores.
	 */
	private static final String QUERY_HANDLER_MINIMAL = "weblab";
	/**
	 * Query handler that retrieve result metadata
	 */
	private static final String QUERY_HANDLER_META = "weblab_meta";
	/**
	 * Query handler that retrieve allow document highlighting
	 */
	private static final String QUERY_HANDLER_HIGHLIGHT = "weblab_highlight";
	/**
	 * Query handler that retrieve facet suggestion
	 */
	private static final String QUERY_HANDLER_FACET = "weblab_facet";
	/**
	 * Query handler that retrieve spell suggestion
	 */
	private static final String QUERY_HANDLER_SPELL = "weblab_spell";
	/**
	 * Query handler that allows "more like this" query
	 */
	private static final String QUERY_HANDLER_MORE = "weblab_more";

	/**
	 * Service ID reference
	 */
	public static final String IDREF = "indexsearch.solr";

	/**
	 * Service ID-res prefix for resultSet
	 */
	public static final String IDRES_RESULT_PREFIX = "result";
	/**
	 * Service ID-res prefix for Query
	 */
	public static final String IDRES_QUERY_PREFIX = "query";
	/**
	 * Service ID-res prefix for Hit
	 */
	public static final String IDRES_HIT_PREFIX = "hit";

	/**
	 * Default SolR prefix
	 */
	public static final String DEFAULT_CORE_PREFIX = "core";

	/**
	 * These are the parameters for the remote Solr server used. As they are
	 * protected, one could change them by overriding the values. However this
	 * is not recommended and the client should use default values.
	 */
	// protected static String SOLR_URL = "http://localhost:8080/solr-weblab/";
	protected static String SOLR_HOME = ".";
	protected static String SOLR_CONFIG = "/conf/solrconfig.xml";
	protected static String SOLR_SCHEMA = "/conf/schema.xml";
	protected static String SOLR_DATA_HOME = SOLR_HOME;

	private static final Log LOGGER = LogFactory.getLog(SolrComponent.class);
	private static final int MIN_TEXT_CONTENT_SIZE = 3;

	private static SolrComponent defaultInstance;
	private static Map<String, SolrComponent> instancesByContext = new HashMap<String, SolrComponent>();

	/**
	 * Retrieve SolrComponent instance depending on context
	 * 
	 * @param context
	 * @return the <code>SolrComponent</solr> associated to the context
	 * @throws WebLabCheckedException
	 */
	public static synchronized SolrComponent getInstance(final String solrURL, final String context) throws WebLabCheckedException {
		if (context == null) {
			if (defaultInstance == null) {
				new SolrComponent(solrURL, null);
			}
			return defaultInstance;
		}

		if (!instancesByContext.containsKey(context)) {
			new SolrComponent(solrURL, context);
		}
		return instancesByContext.get(context);
	}

	public static synchronized void deleteInstance(final String context) {
		throw new NotImplementedException("Instance deletion is not yet implemented. The current solution is to do it manually on the server.");

	}

	/**
	 * Util method to concatenate text content dispatched in multiple Text units
	 * in a document.
	 * 
	 * @param res
	 * @return
	 */
	public static String extractTextFromResource(MediaUnit res) {
		StringBuffer concatenationOfText = new StringBuffer();
		if (res instanceof Document) {
			List<Text> textList = ResourceUtil.getSelectedSubResources(res, Text.class);

			for (Text t : textList) {
				if (t.getContent() != null && t.getContent().length() > MIN_TEXT_CONTENT_SIZE)
					concatenationOfText.append(t.getContent().trim() + "\n");
			}
		} else if (res instanceof Text) {
			Text t = (Text) res;
			if (t.getContent() != null && t.getContent().length() > MIN_TEXT_CONTENT_SIZE)
				concatenationOfText.append(t.getContent().trim() + "\n");
		}
		return concatenationOfText.toString();
	}

	// //////////////////////////////////////////////////////////////////
	// instance part
	// //////////////////////////////////////////////////////////////////

	/**
	 * URL of the remote Solr. This should not change among cores.
	 */
	private String solrURL;

	/**
	 * PATH to the Solr home. This should not change among cores.
	 */
	private String coreDataHome;

	private String coreName;
	private SolrServer server;
	private int nbDocSinceLastFlush = 0;

	/**
	 * Not so default constructor.
	 * 
	 * @param context
	 *            defines the SolR core to be used
	 * @param solrURL
	 *            is the solr server URL
	 * @throws WebLabCheckedException
	 */
	private SolrComponent(String solrURL, String context) throws WebLabCheckedException {
		new SolrComponent(solrURL, SOLR_DATA_HOME, context);
	}

	/**
	 * Extra constructor which enables to used non default parameters.
	 * 
	 * @param context
	 *            which will define the SolR core to be used
	 * @param solrURL
	 *            defining the URL of the remote server
	 * @param solrDataHome
	 *            which locate the folder where data is stored
	 * @throws WebLabCheckedException
	 */
	private SolrComponent(String solrURL, String solrDataHome, String context) throws WebLabCheckedException {
		try {
			// init core variables
			this.coreName = DEFAULT_CORE_PREFIX + (context == null ? "-default" : "" + context.hashCode());
			this.solrURL = solrURL;
			if (!solrDataHome.endsWith(File.separator)) {
				this.coreDataHome = solrDataHome + File.separator + this.coreName + File.separator;
			} else {
				this.coreDataHome = solrDataHome + this.coreName + File.separator;
			}
			// create core (if needed)
			createCore();

			// open solr server to that core
			open();
			LOGGER.info("SolR [" + getCoreName() + "] is ready to serve my lord !");
		} catch (IOException e) {
			throw new WebLabCheckedException("Cannot create SolR [" + getCoreName() + "]. Check the logs", e);
		} catch (SolrServerException e) {
			throw new WebLabCheckedException("Cannot create SolR [" + getCoreName() + "]. Check the logs", e);
		}

		if (context == null) {
			defaultInstance = this;
		} else {
			instancesByContext.put(context, this);
		}
	}

	/**
	 * Check if the current core exists and create it if not
	 * 
	 * @throws SolrServerException
	 *             is thrown when communication with the server does not work
	 * @throws IOException
	 *             n case of incorrect parameters during core creation
	 */
	private void createCore() throws SolrServerException, IOException {
		// test if the core already exists through a get status request
		CoreAdminRequest statusRequest = new CoreAdminRequest();
		statusRequest.setAction(CoreAdminAction.STATUS);
		NamedList<Object> res = new CommonsHttpSolrServer(new URL(solrURL)).request(statusRequest);
		if ((Integer) ((NamedList<Object>) res.get("responseHeader")).get("status") != 0) {
			throw new WebLabUncheckedException("Cannot get Solr server sattus. Check the logs.");
		}
		NamedList<Object> status = (NamedList<Object>) res.get("status");
		NamedList<Object> coreStatus = (NamedList<Object>) status.get(getCoreName());
		if (coreStatus != null) {
			// TODO the core exists do we need to test compatibility or we just
			// use it like that ?
			// LOGGER.debug("SolR [" + getCoreName() +
			// "] already exists. Checking if it matches the current needs...");
			// String instanceDir = coreStatus.get("instanceDir").toString();
			// String dataDir = coreStatus.get("dataDir").toString();
			// if (instanceDir.compareTo(SOLR_HOME) != 0) {
			// throw new WebLabUncheckedException("Cannot use SolR [" +
			// getCoreName() +
			// "]: instanceDir configuration is different from requested one."
			// + " Current is [" + instanceDir + "] vs requested [" +
			// coreDataHome + "].");
			// }
			// if (dataDir.compareTo(getCoreDataHome()) != 0) {
			// throw new WebLabUncheckedException("Cannot use SolR [" +
			// getCoreName() +
			// "]: dataDir configuration is different from requested one."
			// + " Current is [" + dataDir + "] vs requested [" +
			// getCoreDataHome() + "].");
			// }
			LOGGER.debug("Using existing configuration for SolR [" + getCoreName() + "].");
		} else {
			// creation of the core
			LOGGER.debug("Creating SolR [" + getCoreName() + "]...");
			// ask solr for core creation
			Create createRequest = new CoreAdminRequest.Create();
			createRequest.setCoreName(getCoreName());
			createRequest.setInstanceDir(SOLR_HOME);
			createRequest.setDataDir(getCoreDataHome());
			createRequest.setConfigName(SOLR_HOME + SOLR_CONFIG);
			createRequest.setSchemaName(SOLR_HOME + SOLR_SCHEMA);
			res = new CommonsHttpSolrServer(new URL(solrURL)).request(createRequest);

			if ((Integer) ((NamedList<Object>) res.get("responseHeader")).get("status") != 0) {
				throw new WebLabUncheckedException("Cannot create SolR [" + getCoreName() + "]. Check the logs.");
			}
			LOGGER.debug("SolR [" + getCoreName() + "] ready to be used.");
		}
	}

	/**
	 * Create the client for communication with the current core
	 */
	private void open() {
		try {
			LOGGER.debug("Connecting to SOLR server...");
			this.server = new CommonsHttpSolrServer(new URL(getSolrURL() + getCoreName()));
			LOGGER.debug("SolR client is connected to the server.");
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Solr server URL is not valid [" + getSolrURL() + getCoreName() + "].", e);
		}
	}

	/**
	 * Status of SolR core linked to that instance
	 * 
	 * @return some information in JSON-like
	 */
	public String getStatus() {
		CoreAdminRequest statusRequest = new CoreAdminRequest();
		statusRequest.setAction(CoreAdminAction.STATUS);
		statusRequest.setCoreName(getCoreName());
		try {
			NamedList<Object> status = new CommonsHttpSolrServer(new URL(solrURL)).request(statusRequest);

			return status.toString();
		} catch (IOException e) {
			throw new WebLabUncheckedException("Error while requesting SolR [" + getCoreName() + "] status.", e);
		} catch (SolrServerException e) {
			throw new WebLabUncheckedException("Error while requesting SolR [" + getCoreName() + "] status.", e);
		}

	}

	public void addDocument(SolrInputDocument doc) throws WebLabCheckedException {
		if (doc.getField("id") == null) {
			throw new WebLabCheckedException("Document to index does not have valid ID.");
		}
		LOGGER.debug("Indexing document [" + doc.getField("id") + "]...");

		try {
			this.server.add(doc);
			nbDocSinceLastFlush++;
			LOGGER.debug("Resource [" + doc.getField("id") + "] added to the indexing buffer.");
		} catch (SolrException e) {
			throw new WebLabCheckedException("Bad request. SolrDocument does probably not respect the Solr scheme.", e);
		} catch (IOException e) {
			throw new WebLabCheckedException("I/O access error when adding documents", e);
		} catch (SolrServerException e) {
			throw new WebLabCheckedException("Server error while adding documents", e);
		}
	}

	/**
	 * Commit method only flush and optimize index.
	 */
	public void commit() {
		if (this.server != null) {
			try {
				LOGGER.info("Commiting last update on SolR server...");
				if (this.nbDocSinceLastFlush > 0) {
					this.server.commit();
					this.nbDocSinceLastFlush = 0;
					LOGGER.info("Indexing buffer flushed.");
				}
				this.server.optimize();
			} catch (IOException e) {
				throw new WebLabUncheckedException("I/O access error while optimizing the index.", e);
			} catch (SolrServerException e) {
				LOGGER.warn("Cannot optimize the index properly.", e);
			} finally {
				LOGGER.info("Commit done.");
			}
		}
	}

	/**
	 * Search method query solr server with
	 * <ul>
	 * <li>string of the query</li>
	 * <li>offset</li>
	 * <li>limit</li>
	 * </ul>
	 * 
	 * @param queryString
	 *            is the full-text query in Lucene syntax
	 * @param offset
	 *            is the rank of the first result to be returned
	 * @param limit
	 *            is the size of the results list to be returned
	 * @return the Solr <code>QueryResponse</code> containing scored hits each
	 *         one being linked to the Document uri
	 * @throws WebLabCheckedException
	 */
	public QueryResponse search(String queryString, int offset, int limit) throws WebLabCheckedException {
		return search(queryString, offset, limit, null, ORDER.desc);
	}

	/**
	 * Search method query solr server with
	 * <ul>
	 * <li>string of the query</li>
	 * <li>offset</li>
	 * <li>limit</li>
	 * </ul>
	 * 
	 * @param queryString
	 *            is the full-text query in Lucene syntax
	 * @param offset
	 *            is the rank of the first result to be returned
	 * @param limit
	 *            is the size of the results list to be returned
	 * @param sortField
	 *            is a specific field to use for sorting results
	 * @param order
	 *            is the ordering model to use for sorting: either ORDER.desc or
	 *            ORDER.asc
	 * 
	 * @return the Solr <code>QueryResponse</code> containing scored hits each
	 *         one being linked to the Document uri
	 * @throws WebLabCheckedException
	 */
	public QueryResponse search(String queryString, int offset, int limit, String sortField, ORDER order) throws WebLabCheckedException {
		QueryResponse response = new QueryResponse();
		if (!queryString.isEmpty()) {
			SolrQuery query = new SolrQuery();
			query.setQuery(queryString);
			if (sortField != null) {
				query.setSortField(sortField, order);
			}
			query.setParam(CommonParams.START, String.valueOf(offset));
			query.setParam(CommonParams.ROWS, String.valueOf(limit));
			query.setParam(CommonParams.QT, QUERY_HANDLER_MINIMAL);

			// send query to index
			try {
				response = this.server.query(query);
			} catch (SolrServerException sse) {
				LOGGER.error("Cannot post search request", sse);
				throw new WebLabCheckedException("Cannot post search request", sse);
			}
		} else {
			LOGGER.warn("The last query is empty dude ! WHat do you want me to do ?");
		}

		return response;
	}

	/**
	 * Query the solr server to get the metadata associated to a list of
	 * document URIs
	 * 
	 * @param uri
	 *            : the list of URI
	 * @return a solr queryResponse that contain the relevant metadata
	 *         associated to the documents
	 * @throws WebLabCheckedException
	 *             is thrown if the query is not correctly handled
	 */
	public QueryResponse getMetaData(String... uri) throws WebLabCheckedException {
		QueryResponse response = new QueryResponse();
		StringBuilder buf = new StringBuilder();
		buf.append(SolrIndexerConfig.ID_FIELD + ":(");
		String or = " OR ";
		for (int k = 0; k < uri.length; k++) {
			String docURI = uri[k];
			buf.append('\"');
			buf.append(docURI);
			buf.append('\"');
			if (k != uri.length - 1) {
				buf.append(or);
			}
		}
		buf.append(")");

		SolrQuery query = new SolrQuery();
		query.setQuery(buf.toString());

		query.setParam(CommonParams.START, "0");
		query.setParam(CommonParams.ROWS, String.valueOf(uri.length));
		query.setParam(CommonParams.QT, QUERY_HANDLER_META);

		// send the query
		try {
			response = this.server.query(query);
		} catch (SolrServerException sse) {
			LOGGER.error("Cannot post search request", sse);
			throw new WebLabCheckedException("Cannot post search request", sse);
		}

		return response;
	}

	/**
	 * Highlight method that enrich search results with text content where the
	 * matching terms have been emphased.
	 * 
	 * @param queryString
	 *            is the full-text query in Lucene syntax
	 * @param offset
	 *            is the rank of the first result to be returned
	 * @param limit
	 *            is the size of the results list to be returned
	 * @return the Solr <code>QueryResponse</code> containing hits with
	 *         highligted content
	 * @throws WebLabCheckedException
	 */
	public QueryResponse highlight(String queryString, int offset, int limit) throws WebLabCheckedException {
		QueryResponse response = new QueryResponse();

		if (!queryString.isEmpty()) {
			SolrQuery query = new SolrQuery();
			query.setQuery(queryString);
			// query.setSortField("date", ORDER.desc);
			query.setParam(CommonParams.START, String.valueOf(offset));
			query.setParam(CommonParams.ROWS, String.valueOf(limit));
			query.setParam(CommonParams.QT, QUERY_HANDLER_HIGHLIGHT);

			// send query to index
			try {
				response = this.server.query(query);
			} catch (SolrServerException sse) {
				LOGGER.error("Cannot post highlight request", sse);
			}
		} else {
			LOGGER.warn("The input query is empty. Cannot highlight search results. What do you want me to do ?");
		}

		return response;
	}

	public QueryResponse spellSuggest(String queryString) {
		QueryResponse response = new QueryResponse();
		if (!queryString.isEmpty()) {
			SolrQuery query = new SolrQuery();
			query.setQuery(queryString);
			query.setParam(CommonParams.QT, QUERY_HANDLER_SPELL);

			// send query to index
			try {
				response = this.server.query(query);
			} catch (SolrServerException sse) {
				LOGGER.error("Cannot post spell suggest request", sse);
			}
		} else {
			LOGGER.warn("The input query is empty. Cannot make spell suggestion.");
		}
		return response;
	}

	public QueryResponse facetSuggest(String queryString, int offset, int limit) {
		QueryResponse response = new QueryResponse();
		if (!queryString.isEmpty()) {
			SolrQuery query = new SolrQuery();
			query.setQuery(queryString);
			query.setParam(CommonParams.QT, QUERY_HANDLER_FACET);

			// send query to index
			try {
				response = this.server.query(query);
			} catch (SolrServerException sse) {
				LOGGER.error("Cannot post spell suggest request", sse);
			}
		} else {
			LOGGER.warn("The input query is empty. Cannot make spell suggestion.");
		}
		return response;
	}

	/**
	 * MoreLikeThis is a specific Solr query used to retrieve similar documents
	 * from a document id or text. Source, title and text fields are used to
	 * find similar documents.
	 * 
	 * @param queryString
	 *            the reference text or document to find similar docs (id:"doc1"
	 *            for example)
	 * @return the Solr <code>QueryResponse</code> with document list
	 * @throws WebLabCheckedException
	 */
	public QueryResponse moreLikeThis(String queryString, int offset, int limit) throws WebLabCheckedException {
		QueryResponse response = new QueryResponse();
		if (!queryString.isEmpty()) {
			SolrQuery query = new SolrQuery();
			query.setParam(CommonParams.QT, QUERY_HANDLER_MORE);
			query.setParam(CommonParams.START, String.valueOf(offset));
			query.setParam(CommonParams.ROWS, String.valueOf(limit));

			// query.setQueryType(MoreLikeThisParams.MLT);
			query.set(MoreLikeThisParams.MLT);
			query.set(MoreLikeThisParams.MATCH_INCLUDE, false);
			query.set(MoreLikeThisParams.MIN_DOC_FREQ, 1);
			query.set(MoreLikeThisParams.MIN_TERM_FREQ, 1);
			query.set(MoreLikeThisParams.SIMILARITY_FIELDS, "source,title,text");
			query.setQuery(queryString);

			// Send query to index
			try {
				response = this.server.query(query);
			} catch (SolrServerException sse) {
				LOGGER.error("Cannot post search request", sse);
				throw new WebLabCheckedException("Cannot post search request", sse);
			}
		} else {
			LOGGER.warn("The input query is empty. Cannot make a 'moreLikeThis' query.");
		}

		return response;
	}

	/**
	 * Ask for document deletion based on docurment URI
	 * 
	 * @param docURI
	 *            the URI of the document to delete.
	 * @return UpdateResponse : empty object just to check the deletion was
	 *         done.
	 * @throws WebLabCheckedException
	 *             in case of any problem.
	 */
	public UpdateResponse deleteDocbyURI(String docURI) throws WebLabCheckedException {
		StringBuffer query = new StringBuffer(SolrIndexerConfig.ID_FIELD);
		query.append(':');
		query.append('"');
		query.append(docURI);
		query.append('"');

		return deleteDocbyQuery(query.toString());
	}

	/**
	 * Ask for document deletion based on a query: all documents matching the
	 * query will be deleted.
	 * 
	 * @param query
	 *            a query to select documents
	 * @return UpdateResponse : empty object just to check the deletion was
	 *         done.
	 * @throws WebLabCheckedException
	 *             in case of any problem.
	 */
	public UpdateResponse deleteDocbyQuery(String query) throws WebLabCheckedException {
		UpdateResponse response = new UpdateResponse();
		try {
			response = this.server.deleteByQuery(query.toString());
		} catch (SolrServerException sse) {
			LOGGER.error("Request to delete failed for query [" + query + "].", sse);
			throw new WebLabCheckedException("Delete failed", sse);
		} catch (IOException e) {
			LOGGER.error("Request to delete failed for query [" + query + "].", e);
			throw new WebLabCheckedException("Delete failed", e);
		}
		return response;
	}

	public String getCoreName() {
		return coreName;
	}

	public String getSolrURL() {
		return solrURL;
	}

	public void setSolrURL(String solrURL) {
		if (!solrURL.endsWith(File.separator)) {
			this.solrURL = solrURL + File.separator;
		} else {
			this.solrURL = solrURL;
		}
	}

	public String getCoreDataHome() {
		return coreDataHome;
	}

	public void setCoreDataHome(String coreDataHome) {
		coreDataHome = new File(coreDataHome).getAbsolutePath();
		if (!coreDataHome.endsWith(File.separator)) {
			this.coreDataHome = coreDataHome + File.separator;
		} else {
			this.coreDataHome = coreDataHome;
		}
	}
}
