package org.ow2.weblab.services.solr.analyser;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.model.retrieval.WRetrievalAnnotator;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.searcher.SolrSearcher;
import org.ow2.weblab.util.index.Field;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class FacetSuggestion implements Analyser {
	public static final String BEAN_NAME = "facetSuggestionServiceBean";
	public static final String IDRES_FACET_PREFIX = "facets";

	private static long suggestionCounter = 0;
	private static long queryCounter = 0;

	private Log logger;
	private String solrURL;
	private Set<Field> facetFields;

	private boolean noCore = false;

	@PostConstruct
	public void init() {
		this.logger = LogFactory.getLog(SolrSearcher.class);
		try {
			new URL(solrURL);
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + solrURL + "].", e);
		}
	}

	@PreDestroy
	public void destroy() {
		logger.info("Destroying SolR FacetSuggestion service.");

		// nothing to do I guess

	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, UnexpectedException, InvalidParameterException, ContentNotAvailableException,
			InsufficientResourcesException, UnsupportedRequestException, ServiceNotConfiguredException {

		checkArgs(args);

		ResultSet enrichSet = doFacetSuggest(args.getUsageContext(), (ResultSet) args.getResource());

		ProcessReturn ret = new ProcessReturn();
		ret.setResource(enrichSet);
		return ret;
	}

	public ResultSet doFacetSuggest(String usageContext, ResultSet set) {

		try {
			if (!(set.getResource().get(0) instanceof StringQuery)) {
				throw new WebLabCheckedException("There is no query in the ResultSet.");
			}
			StringQuery query = (StringQuery) set.getResource().get(0);
			WRetrievalAnnotator wra = new WRetrievalAnnotator(new URI(set.getUri()), set.getPok());

			if (wra.readNumberOfResults().firstTypedValue() == 0) {
				throw new WebLabCheckedException("There are no results in the ResultSet.");
			}

			int offset = wra.readExpectedOffset().firstTypedValue();
			int limit = wra.readHit().size();

			SolrComponent instance;
			if (noCore) {
				instance = SolrComponent.getInstance(solrURL, null);
			} else {
				instance = SolrComponent.getInstance(solrURL, usageContext);
			}
			QueryResponse response = instance.facetSuggest(query.getRequest(), offset, limit);

			if (response.getFacetFields() == null || response.getFacetFields().size() == 0) {
				throw new WebLabCheckedException("There are no facets in the ResultSet.");
			} else {
				ComposedResource cres = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_FACET_PREFIX + suggestionCounter++, ComposedResource.class);
				set.getResource().add(cres);
				for (FacetField facet : response.getFacetFields()) {
					final List<Count> countList = facet.getValues();
					if (countList == null) {
						logger.debug("Facet " + facet.getName() + " is empty.");
					} else {
						for (final Count valuedFacet : countList) {
							int nbRes = (int) valuedFacet.getCount();
							String facetFilter = valuedFacet.getAsFilterQuery();

							StringQuery facetQuery = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_FACET_PREFIX + SolrComponent.IDRES_QUERY_PREFIX
									+ queryCounter++, StringQuery.class);
							facetQuery.setRequest(query.getRequest() + ' ' + facetFilter);

							WRetrievalAnnotator wra4q = new WRetrievalAnnotator(facetQuery);
							wra4q.writeNumberOfResults(nbRes);
							wra4q.writeExpectedOffset(offset);
							wra4q.writeExpressedWith(this.getClass().getSimpleName());
							
							// TODO use specific properties
							wra4q.writeLabel(facet.getName());
							wra4q.writeDescription(facetFilter);

							cres.getResource().add(facetQuery);
						}
					}
				}
			}
			logger.info("Facet suggestion done for ["+set.getUri()+"].");
		} catch (WebLabCheckedException e) {
			logger.info("Cannot make spell suggestion : " + e.getMessage());
		} catch (URISyntaxException e) {
			logger.info("Cannot make spell suggestion : " + e.getMessage());
		}
		return set;
	}

	private void checkArgs(ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("Input args for [" + this.getClass().getSimpleName() + "] cannot be null.");
		}
		// TODO finish it...
	}

	public String getSolrURL() {
		return solrURL;
	}

	public void setSolrURL(String solrURL) {
		this.solrURL = solrURL;
	}

	public Set<Field> getFacetFields() {
		return facetFields;
	}

	public void setFacetFields(Set<Field> facetFields) {
		this.facetFields = facetFields;
	}

	public boolean isNoCore() {
		return noCore;
	}

	public void setNoCore(boolean noCore) {
		this.noCore = noCore;
	}

}
