package org.ow2.weblab.services.solr.analyser;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.retrieval.WRetrievalAnnotator;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.rdf.Value;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.indexer.SolrIndexerConfig;
import org.ow2.weblab.services.solr.searcher.SolrSearcher;
import org.ow2.weblab.util.index.Field;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class ResultSetMetadataEnrichment implements Analyser {
	public static final String BEAN_NAME = "resultSetMetaEnricherServiceBean";

	private Log logger;
	private String solrURL;
	private List<Field> fields;
	private SimpleDateFormat dateFormat;

	private boolean noCore = false;

	@PostConstruct
	public void init() {
		this.logger = LogFactory.getLog(SolrSearcher.class);
		// init date formatter
		this.dateFormat = new SimpleDateFormat(SolrIndexerConfig.SOLR_DATE_FORMAT);
		try {
			new URL(solrURL);
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + solrURL + "].", e);
		}
	}

	@PreDestroy
	public void destroy() {
		// nothing to do I guess
	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, UnexpectedException, InvalidParameterException, ContentNotAvailableException,
			InsufficientResourcesException, UnsupportedRequestException, ServiceNotConfiguredException {

		checkArgs(args);

		ResultSet enrichSet = addMetadataToResultSet(args.getUsageContext(), (ResultSet) args.getResource());

		ProcessReturn ret = new ProcessReturn();

		ret.setResource(enrichSet);
		return ret;
	}

	private void checkArgs(ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("Input args for [" + this.getClass().getSimpleName() + "] cannot be null.");
		}
		// TODO finish it...
	}

	public ResultSet addMetadataToResultSet(String usageContext, ResultSet resultSet) throws UnexpectedException {
		try {
			WRetrievalAnnotator wra = new WRetrievalAnnotator(new URI(resultSet.getUri()), resultSet.getPok());

			JenaPoKHelper hlpr = new JenaPoKHelper(resultSet.getPok());
			hlpr.setAutoCommitMode(false);

			List<URI> docUris = new ArrayList<URI>();
			if (wra.readHit() == null || wra.readHit().size() == 0) {
				logger.debug("No hit in the resultSet.");
			} else {
				Value<URI> hits = wra.readHit();
				for (URI hit : hits) {
					wra.startInnerAnnotatorOn(hit);
					docUris.addAll(wra.readLinkedTo().getValues());
					wra.endInnerAnnotator();
				}
				String[] uris = new String[docUris.size()];
				for (int k = 0; k < docUris.size(); k++) {
					uris[k] = docUris.get(k).toString();
				}
				SolrComponent instance;
				if(noCore){
					instance = SolrComponent.getInstance(solrURL, null);
				}else{
					instance = SolrComponent.getInstance(solrURL, usageContext);
				}
				QueryResponse response = instance.getMetaData(uris);

				if (response.getResults() == null || response.getResults().size() == 0) {
					throw new WebLabCheckedException("Cannot get metadata from Solr.");
				} else {
					if (response.getResults().size() < docUris.size()) {
						logger.info("Solr returned only " + response.getResults().size() + "... We'll do what we can.");
					}

					for (SolrDocument doc : response.getResults()) {
						for (Field field : fields) {
							addFieldDataToPok(hlpr, (String) doc.getFieldValue(SolrIndexerConfig.ID_FIELD), doc, field);
						}
					}
				}
				hlpr.commit();
			}
			logger.info("Metadata enrichment done for ["+resultSet.getUri()+"].");
		} catch (URISyntaxException e) {
			throw new WebLabUncheckedException("Some URI were invalid during annotation. This should not happend really...", e);
		} catch (WebLabCheckedException e) {
			throw new UnexpectedException("Well metadata enrichment failed: " + e.getMessage(), e);
		}
		return resultSet;
	}

	private void addFieldDataToPok(JenaPoKHelper hlpr, String docURI, SolrDocument doc, Field field) throws WebLabCheckedException {
		if (field.getProperties().size() == 1) {
			if (doc.getFieldValue(field.getName()) == null) {
				logger.debug("Cannot get metadata from field [" + field.getName() + "]. It does not contain anything.");
			} else {
				for (Object value : doc.getFieldValues(field.getName())) {
					switch (field.getType()) {
					case DATE:
						Date d = (Date) value;
						hlpr.createLitStat(docURI, field.getProperties().get(0).trim(), dateFormat.format(d));
						break;
					case RES_URI:
						hlpr.createResStat(docURI, field.getProperties().get(0).trim(), value.toString());
						break;
					// case URI:
					// case LONG:
					// case TEXT:
					default:
						hlpr.createLitStat(docURI, field.getProperties().get(0).trim(), value.toString());
					}
				}
			}
		} else {
			// that should be a text only field or a field that merge multiple
			// properties => it should not be in the list for metadata
			// enrichment.
			throw new WebLabCheckedException("Cannot get metadata from field [" + field.getName()
					+ "]. It should not be in the metadata enrichmenet field list.");
		}
	}

	public String getSolrURL() {
		return solrURL;
	}

	public void setSolrURL(String solrURL) {
		this.solrURL = solrURL;
	}

	public List<Field> getFields() {
		return fields;
	}

	public void setFields(List<Field> fields) {
		this.fields = fields;
	}

	public boolean isNoCore() {
		return noCore;
	}

	public void setNoCore(boolean noCore) {
		this.noCore = noCore;
	}

}
