package org.ow2.weblab.services.solr.analyser;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.common.SolrDocument;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.JenaResourceHelper;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.retrieval.WRetrievalAnnotator;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.SolrConfig;
import org.ow2.weblab.util.SolrQueryParser;
import org.ow2.weblab.util.WebLabQueryParser;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class Highlighter implements Analyser {
	private static final int DEFAULT_SNIPPET_SIZE = 300;

	public static final String BEAN_NAME = "highlighterServiceBean";

	private Log logger;
	private SolrConfig conf;
	private WebLabQueryParser parser;

	@PostConstruct
	public void init() {
		this.logger = LogFactory.getLog(Highlighter.class);
		this.parser = new SolrQueryParser(conf);
		try {
			new URL(conf.getSolrURL());
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + conf.getSolrURL() + "].", e);
		}
	}

	@PreDestroy
	public void destroy() {
		logger.info("Destroying SolR Highlighter service.");

		// nothing to do I guess

	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, UnexpectedException, InvalidParameterException, ContentNotAvailableException,
			InsufficientResourcesException, UnsupportedRequestException, ServiceNotConfiguredException {
		checkArgs(args);

		ResultSet enrichSet = highLightHitInResultSet(args.getUsageContext(), (ResultSet) args.getResource());

		ProcessReturn ret = new ProcessReturn();

		ret.setResource(enrichSet);
		return ret;
	}

	public ResultSet highLightHitInResultSet(String usageContext, ResultSet set) {

		try {
			if (!(set.getResource().get(0) instanceof Query)) {
				throw new WebLabCheckedException("There is no query in the ResultSet.");
			}
			Query query = (Query) set.getResource().get(0);
			WRetrievalAnnotator wra = new WRetrievalAnnotator(new URI(set.getUri()), set.getPok());

			if (wra.readNumberOfResults().firstTypedValue() == 0) {
				throw new WebLabCheckedException("There are no results in the ResultSet.");
			}

			int offset = wra.readExpectedOffset().firstTypedValue();
			int limit = wra.readHit().size();

			SolrComponent instance;
			if (conf.isNoCore()) {
				instance = SolrComponent.getInstance(conf, null);
			} else {
				instance = SolrComponent.getInstance(conf, usageContext);
			}

			// check if the query has annotation to expect a specific behavior
			boolean ascOrder = false;
			String orderedByProperty = null;
			if (query.getAnnotation().size() > 0) {
				ResourceHelper hlpr = new JenaResourceHelper(query);
				ascOrder = parser.getOrder(query, hlpr);
				orderedByProperty = parser.getOrderBy(query, hlpr);
			}

			QueryResponse response = instance.highlight(parser.getRequest(query, new JenaResourceHelper(query)), offset, limit, orderedByProperty, ascOrder ? ORDER.asc : ORDER.desc);

			Map<String, Map<String, List<String>>> highlights = response.getHighlighting();
			JenaPoKHelper hlpr = new JenaPoKHelper(set.getPok(), false);
			for (SolrDocument hit : response.getResults()) {
				String resourceUri = String.valueOf(hit.getFieldValue(SolrConfig.FIELD_ID));

				Set<String> hits = hlpr.getSubjsOnPredRes(WebLabRetrieval.IS_LINKED_TO, resourceUri);
				if (hits.size() != 1) {
					logger.info("Well, the resultSet is no clear: multiple hits (or none) are linked to [" + resourceUri + "]. Let's try to cope with that...");
				}
				for (String hitUri : hits) {
					if (highlights.get(hit.getFieldValue(SolrConfig.FIELD_ID)).size() > 0
							&& highlights.get(hit.getFieldValue(SolrConfig.FIELD_ID)) != null
							&& highlights.get(hit.getFieldValue(SolrConfig.FIELD_ID)).get(SolrConfig.FIELD_PLAIN_TEXT) != null) {
						List<String> hitHighlightingList = highlights.get(hit.getFieldValue(SolrConfig.FIELD_ID)).get(SolrConfig.FIELD_PLAIN_TEXT);
						
						StringBuffer snippet = new StringBuffer();
						for (String hl : hitHighlightingList) {
							snippet.append(hl);
						}
						hlpr.createLitStat(hitUri, WebLabRetrieval.HAS_DESCRIPTION, snippet.toString());
					} else {
						logger.debug("No highlighted content available for hit [" + hitUri + "] on document [" + resourceUri + "]. Trying to use the "
								+ DEFAULT_SNIPPET_SIZE + " first characters of content as snippet.");
						if (hit.getFieldNames().contains(SolrConfig.FIELD_PLAIN_TEXT)) {

							String content = hit.get(SolrConfig.FIELD_PLAIN_TEXT).toString();
							content = content.substring(0, Math.min(content.length(), DEFAULT_SNIPPET_SIZE)).trim() + "...";
							hlpr.createLitStat(hitUri, WebLabRetrieval.HAS_DESCRIPTION, content);
						} else {
							logger.warn("No highlighted content for hit [" + hitUri + "] on document [" + resourceUri
									+ "] and text not available... we can't do nothing !");
						}
					}
				}
			}
			hlpr.commit();
			logger.info("Highlighting done for [" + set.getUri() + "].");

			return set;
		} catch (WebLabCheckedException e) {
			logger.info("Cannot highlight results: " + e.getMessage(), e);
		} catch (URISyntaxException e) {
			logger.info("Cannot highlight results: " + e.getMessage(), e);
		}
		return set;
	}

	private void checkArgs(ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("Input args for [" + this.getClass().getSimpleName() + "] cannot be null.");
		}
		// TODO finish it...
	}

	public SolrConfig getConf() {
		return conf;
	}

	public void setConf(SolrConfig conf) {
		this.conf = conf;
	}
}