package org.ow2.weblab.services.solr.analyser;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse;
import org.apache.solr.client.solrj.response.SpellCheckResponse.Suggestion;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.SolrConfig;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class SpellSuggestion implements Analyser {
	public static final String BEAN_NAME = "spellSuggestionServiceBean";

	private static final String IDRES_SPELL_PREFIX = "spell";

	private static long suggestionCounter = 0;
	private static long queryCounter = 0;

	private Log logger;
	private SolrConfig conf;
	private int nbSuggestion = 10;

	@PostConstruct
	public void init() {
		this.logger = LogFactory.getLog(SpellSuggestion.class);
		try {
			new URL(conf.getSolrURL());
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + conf.getSolrURL() + "].", e);
		}
	}

	@PreDestroy
	public void destroy() {
		logger.info("Destroying SolR SpellSuggestion service.");

		// nothing to do I guess

	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, UnexpectedException, InvalidParameterException, ContentNotAvailableException,
			InsufficientResourcesException, UnsupportedRequestException, ServiceNotConfiguredException {
		checkArgs(args);

		ResultSet enrichSet = doSpellSuggest(args.getUsageContext(), (ResultSet) args.getResource());

		ProcessReturn ret = new ProcessReturn();

		ret.setResource(enrichSet);
		return ret;
	}

	public ResultSet doSpellSuggest(String usageContext, ResultSet set) {

		try {
			if (!(set.getResource().get(0) instanceof StringQuery)) {
				throw new WebLabCheckedException("There is no query in the ResultSet.");
			}
			StringQuery q = (StringQuery) set.getResource().get(0);
			String originalQuery = q.getRequest();
			SolrComponent instance;
			if(conf.isNoCore()){
				instance = SolrComponent.getInstance(conf, null);
			}else{
				instance = SolrComponent.getInstance(conf, usageContext);
			}
			QueryResponse response = instance.spellSuggest(q.getRequest());
			SpellCheckResponse scr = response.getSpellCheckResponse();

			if (scr == null || scr.isCorrectlySpelled()) {
				logger.debug("The query in the ResultSetis correctly spelled.");
			} else {
				logger.debug("Getting spell check suggestions");
				Set<StringQuery> suggestedQueries = new HashSet<StringQuery>();

				Map<String, Suggestion> mscr = scr.getSuggestionMap();
				// testing if there is term suggested
				if (mscr.size() == 0) {
					logger.debug("The query isn't correctly spelled, but we don't have any suggestion.");
				} else if (mscr.size() == 1) {
					// if so then produce the suggested requests simply
					String s = mscr.entrySet().iterator().next().getKey();
					Suggestion suggestion = mscr.get(s);
					List<String> ls = suggestion.getAlternatives();
					List<Integer> fs = suggestion.getAlternativeFrequencies();
					for (int i = 0; i < ls.size(); i++) {
						StringQuery str = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_SPELL_PREFIX + SolrComponent.IDRES_QUERY_PREFIX
								+ queryCounter++, StringQuery.class);
						str.setRequest(originalQuery.replace(originalQuery.substring(suggestion.getStartOffset(), suggestion.getEndOffset()), ls.get(i)));
						str.setWeight(fs.get(i));
						logger.debug("Adding spell check suggestion [" + str.getRequest() + "]@" + str.getWeight());
						suggestedQueries.add(str);
					}
				} else {
					// if not we need to produce all possible queries and then
					// weight them to get the top N suggestions
					StringQuery[] topNSuggestedQueries = new StringQuery[nbSuggestion];
					double[] topN = new double[nbSuggestion];
					for (String s : mscr.keySet()) {
						Suggestion suggestion = mscr.get(s);
						List<String> ls = suggestion.getAlternatives();
						List<Integer> fs = suggestion.getAlternativeFrequencies();

						Set<StringQuery> suggestedQueriesToBeAdded = new HashSet<StringQuery>();
						for (int i = 0; i < ls.size(); i++) {
							if (suggestedQueries.size() > 0) {

								for (StringQuery query : suggestedQueries) {
									StringQuery str = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_SPELL_PREFIX + SolrComponent.IDRES_QUERY_PREFIX
											+ queryCounter++, StringQuery.class);
									// compile the request
									str.setRequest(query.getRequest().replace(suggestion.getToken(), ls.get(i)));
									// average the weight
									str.setWeight((fs.get(i) + query.getWeight()) / 2);
									suggestedQueriesToBeAdded.add(str);
									addTopSocredSuggestion(topN, topNSuggestedQueries, str);
								}

							}

							StringQuery str = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_SPELL_PREFIX + SolrComponent.IDRES_QUERY_PREFIX
									+ queryCounter++, StringQuery.class);
							str.setRequest(originalQuery.replace(suggestion.getToken(), ls.get(i)));
							str.setWeight(fs.get(i));
							suggestedQueries.add(str);
							logger.debug("Adding spell check suggestion [" + str.getRequest() + "]@" + str.getWeight());

							addTopSocredSuggestion(topN, topNSuggestedQueries, str);
						}
						if (suggestedQueriesToBeAdded.size() > 0) {
							suggestedQueries.addAll(suggestedQueriesToBeAdded);
						}
					}
				}
				if (suggestedQueries == null || suggestedQueries.size() == 0) {
					logger.debug("There is no spell check suggestions done despite the suggested terms...");
				} else {
					logger.debug("Adding spell check suggestions to result set.");
					ComposedResource collec = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_SPELL_PREFIX + suggestionCounter++,
							ComposedResource.class);
					collec.getResource().addAll(suggestedQueries);
					set.getResource().add(collec);
				}
			}
			logger.info("Spell suggestion done for ["+set.getUri()+"].");
		} catch (WebLabCheckedException e) {
			logger.warn("Cannot make spell suggestion : " + e.getMessage());
		}
		return set;
	}

	private void addTopSocredSuggestion(double[] topN, StringQuery[] topNSuggestedQueries, StringQuery str) {
		for (int i = 0; i < topN.length; i++) {
			if (topN[i] < str.getWeight()) {
				topN[i] = str.getWeight();
				topNSuggestedQueries[i] = str;
			}
		}
	}

	private void checkArgs(ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("Input args for [" + this.getClass().getSimpleName() + "] cannot be null.");
		}
		// TODO finish it...
	}

	public int getNbSuggestion() {
		return nbSuggestion;
	}

	public void setNbSuggestion(int nbSuggestion) {
		this.nbSuggestion = nbSuggestion;
	}

	public SolrConfig getConf() {
		return conf;
	}

	public void setConf(SolrConfig conf) {
		this.conf = conf;
	}

}
