package org.ow2.weblab.util;

import java.util.HashSet;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.model.SimilarityQuery;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.services.solr.SolrConfig;
import org.ow2.weblab.services.solr.searcher.SolrSearcher;
import org.ow2.weblab.util.index.Field;

public class SolrQueryParser extends WebLabQueryParser {
	private Log logger = LogFactory.getLog(SolrQueryParser.class);

	private SolrConfig config;

	public SolrQueryParser(SolrConfig config) {
		super();
		this.config = config;
	}

	@Override
	public String getRequest(Query q, ResourceHelper hlpr) throws WebLabCheckedException {
		if (q instanceof StringQuery) {
			return ((StringQuery) q).getRequest();
		} else if (q instanceof ComposedQuery) {
			return getComposedRequest((ComposedQuery) q, hlpr);
		} else if (q instanceof SimilarityQuery) {
			SimilarityQuery query = (SimilarityQuery) q;
			StringBuffer queryString = new StringBuffer();
			List<Resource> resources = query.getResource();

			if (resources.size() == 1) {
				// single sample => can use more like this function
				queryString.append(SolrConfig.FIELD_ID);
				queryString.append(':');
				queryString.append('"');
				queryString.append(resources.get(0).getUri());
				queryString.append('"');
			} else {
				// multiple samples, thus try simple ID based query
				// TODO check this is the expected behavior
				for (int i = 0; i < resources.size(); i++) {
					Resource r = resources.get(i);
					if (i > 0) {
						queryString.append(" OR ");
					}
					queryString.append(SolrConfig.FIELD_ID);
					queryString.append(':');
					queryString.append('"');
					queryString.append(r.getUri());
					queryString.append('"');
				}
			}
			return queryString.toString();
		}
		throw new WebLabCheckedException("Cannot parse query type : " + q.getClass() + ".");
	}

	@Override
	public String getComposedRequest(ComposedQuery q, ResourceHelper hlpr) throws WebLabCheckedException {
		// Watch out : this supposes that WebLab Operator are following Lucene
		// operator syntax...
		String operator = q.getOperator().value();

		StringBuffer queryString = new StringBuffer();
		queryString.append('(');
		for (Query subQ : q.getQuery()) {
			if (queryString.length() > 1) {
				queryString.append(' ');
				queryString.append(operator);
				queryString.append(' ');
			}
			if (subQ instanceof StringQuery) {
				queryString.append(getRequestWithScope((StringQuery) subQ, hlpr));
			} else if (subQ instanceof ComposedQuery) {
				queryString.append(getComposedRequest((ComposedQuery) subQ, hlpr));
			} else {
				throw new WebLabCheckedException("Solr engine service can only process " + ComposedQuery.class.getSimpleName() + " or "
						+ StringQuery.class.getSimpleName() + ".");
			}
		}
		queryString.append(')');
		return queryString.toString();
	}

	@Override
	public String getRequestWithScope(Query q, ResourceHelper hlpr) throws WebLabCheckedException {
		List<String> values = hlpr.getRessOnPredSubj(q.getUri(), SolrSearcher.HAS_SCOPE);
		if (values.size() == 0) {
			logger.debug("Query has no scope.");
		} else if (values.size() > 1 && new HashSet<String>(values).size() > 1) {
			logger.info("Query is holding multiple values for [" + SolrSearcher.HAS_SCOPE + "]. That's not cool, so we ignore all of them.");
		} else if (!config.getPropertyToFieldMap().containsKey(values.get(0))) {
			logger.warn("The value for [" + SolrSearcher.HAS_SCOPE + "] is [" + values.get(0)
					+ "] which is unknown from SolrIndexerConfig. That's not cool, so we ignore it.");
		} else {
			Field f = config.getPropertyToFieldMap().get(values.get(0));
			return f.getName() + SolrConfig.FIELD_SEPARATOR + "(" + getRequest(q, hlpr) + ")";
		}
		return getRequest(q, hlpr);
	}

	@Override
	public String getOrderBy(Query q, ResourceHelper hlpr) throws WebLabCheckedException {
		List<String> values = hlpr.getRessOnPredSubj(q.getUri(), SolrSearcher.TO_BE_RANKED_BY);
		if (values.size() == 0) {
			logger.debug("Query has no 'order by' part.");
		} else if (values.size() > 1 && new HashSet<String>(values).size() > 1) {
			logger.info("Query is holding multiple values for [" + SolrSearcher.TO_BE_RANKED_BY + "]. That's not funky, so we ignore all of them.");
		} else {
			if (config != null) {
				return config.getPropertyToFieldMap().get(values.get(0)).getName();
			}
		}
		return null;
	}

	@Override
	public boolean getOrder(Query q, ResourceHelper hlpr) throws WebLabCheckedException {
		List<String> values = hlpr.getLitsOnPredSubj(q.getUri(), SolrSearcher.TO_BE_RANKED_ASCENDING);
		if (values.size() == 0) {
			logger.debug("Query has no ordering specified.");
		} else if (values.size() > 1 && new HashSet<String>(values).size() > 1) {
			logger.info("Query is holding multiple values for [" + SolrSearcher.TO_BE_RANKED_ASCENDING
					+ "]. That's not well perceived, so we ignore all of them.");
		} else {
			return Boolean.parseBoolean(values.get(0));
		}
		return false;
	}

}
