package org.ow2.weblab.services.solr.analyser;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.solr.client.solrj.SolrQuery.ORDER;
import org.apache.solr.client.solrj.response.FacetField;
import org.apache.solr.client.solrj.response.FacetField.Count;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.factory.AnnotationFactory;
import org.ow2.weblab.core.extended.factory.ResourceFactory;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabRetrieval;
import org.ow2.weblab.core.helper.ResourceHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.helper.impl.JenaResourceHelper;
import org.ow2.weblab.core.model.ComposedQuery;
import org.ow2.weblab.core.model.ComposedResource;
import org.ow2.weblab.core.model.Operator;
import org.ow2.weblab.core.model.Query;
import org.ow2.weblab.core.model.ResultSet;
import org.ow2.weblab.core.model.StringQuery;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;
import org.ow2.weblab.services.solr.SolrComponent;
import org.ow2.weblab.services.solr.SolrConfig;
import org.ow2.weblab.services.solr.searcher.SolrSearcher;
import org.ow2.weblab.util.SolrQueryParser;
import org.ow2.weblab.util.WebLabQueryParser;
import org.ow2.weblab.util.index.Field;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class FacetSuggestion implements Analyser {

	private static final String FILTER_TAG = "/filter";
	public static final String BEAN_NAME = "facetSuggestionServiceBean";
	public static final String IDRES_FACET_PREFIX = "facets";

	private static long suggestionCounter = 0;
	private static long queryCounter = 0;

	private Log logger;
	private SolrConfig conf;
	private Set<Field> facetFields;

	private Map<String, String> fieldToPropertyMap = new HashMap<String, String>();
	private WebLabQueryParser parser;

	@PostConstruct
	public void init() {
		this.logger = LogFactory.getLog(FacetSuggestion.class);
		try {
			new URL(conf.getSolrURL());
		} catch (MalformedURLException e) {
			throw new WebLabUncheckedException("Cannot start the service. The solrULR is invalid [" + conf.getSolrURL() + "].", e);
		}
		parser = new SolrQueryParser(conf);

		for (Field f : facetFields) {
			if (f.getProperties() != null && f.getProperties().size() > 0) {
				// TODO find another way since it does not take into account the
				// mapping from field to property which is 1..*

				if (f.getEntityTypes() != null && f.getEntityTypes().size() > 0) {
					// use entity type filter as scope
					fieldToPropertyMap.put(f.getName(), f.getEntityTypes().get(0));
				} else {
					// use property filter as scope
					fieldToPropertyMap.put(f.getName(), f.getProperties().get(0));
				}
			}
		}
	}

	@PreDestroy
	public void destroy() {
		logger.info("Destroying SolR FacetSuggestion service.");

		// nothing to do I guess

	}

	@Override
	public ProcessReturn process(ProcessArgs args) throws AccessDeniedException, UnexpectedException, InvalidParameterException, ContentNotAvailableException,
			InsufficientResourcesException, UnsupportedRequestException, ServiceNotConfiguredException {

		checkArgs(args);

		ResultSet enrichSet = doFacetSuggest(args.getUsageContext(), (ResultSet) args.getResource());

		ProcessReturn ret = new ProcessReturn();
		ret.setResource(enrichSet);
		return ret;
	}

	public ResultSet doFacetSuggest(String usageContext, ResultSet set) {

		try {
			if (!(set.getResource().get(0) instanceof Query)) {
				throw new WebLabCheckedException("There is no query in the ResultSet.");
			}
			Query query = (Query) set.getResource().get(0);
			ComposedResource cres = doFacetSuggest(usageContext, query);
			set.getResource().add(cres);
		} catch (WebLabCheckedException e) {
			logger.info("Cannot make spell suggestion : " + e.getMessage());
		}
		return set;
	}

	public ComposedResource doFacetSuggest(String usageContext, Query query) {
		ComposedResource cres = ResourceFactory.createResource(SolrComponent.IDREF, IDRES_FACET_PREFIX + suggestionCounter++, ComposedResource.class);
		try {
			SolrComponent instance;
			if (conf.isNoCore()) {
				instance = SolrComponent.getInstance(conf, null);
			} else {
				instance = SolrComponent.getInstance(conf, usageContext);
			}

			// check if the query has annotation to expect a specific behavior
			boolean ascOrder = false;
			String orderedByProperty = null;
			if (query.getAnnotation().size() > 0) {
				ResourceHelper hlpr = new JenaResourceHelper(query);
				ascOrder = parser.getOrder(query, hlpr);
				orderedByProperty = parser.getOrderBy(query, hlpr);
			}
			String request = parser.getRequest(query, new JenaResourceHelper(query));
			QueryResponse response = instance.facetSuggest(request, 0, 1, orderedByProperty,
					ascOrder ? ORDER.asc : ORDER.desc);

			if (response.getFacetFields() == null || response.getFacetFields().size() == 0) {
				throw new WebLabCheckedException("There are no facets in the ResultSet.");
			} else {

				for (FacetField facet : response.getFacetFields()) {
					final List<Count> countList = facet.getValues();
					if (countList == null) {
						logger.debug("Facet " + facet.getName() + " is empty.");
					} else {
						for (final Count valuedFacet : countList) {
							int nbRes = (int) valuedFacet.getCount();
							if (nbRes == 0) {
								logger.debug("Facet " + facet.getName() + ":" + valuedFacet.getName() + " is empty.");
							} else {
								Query facetQuery = createFacetQuery(query, 0, facet, nbRes, valuedFacet.getName());
								cres.getResource().add(facetQuery);
							}
						}
					}
				}
			}
			logger.info("Facet suggestion done for [" + query.getUri() + "].");
		} catch (WebLabCheckedException e) {
			logger.info("Cannot make spell suggestion : " + e.getMessage());
		}
		return cres;
	}

	/**
	 * Create facet query in accordance to the new usage of StringQuery => issue
	 * WEBLAB-316
	 * 
	 * @param query
	 *            is the original query that was sent to generate the facets
	 * @param offset
	 *            is the original offset of the query
	 * @param facet
	 *            is the fact as a FacetField
	 * @param nbRes
	 *            is the number of results associated to that facet if applied
	 *            as filter
	 * @param facetFilter
	 *            is the filter to apply as a String
	 * @return a StringQuery with all information related to scope and link in
	 *         an annotation.
	 */
	private Query createFacetQuery(Query query, int offset, FacetField facet, int nbRes, String facetFilter) {
		String cQueryIdRes = IDRES_FACET_PREFIX + SolrComponent.IDRES_QUERY_PREFIX + queryCounter++;

		// create facet filter
		StringQuery facetQuery = ResourceFactory.createResource(SolrComponent.IDREF, cQueryIdRes + FILTER_TAG, StringQuery.class);
		// add scope on query to filter on specific field based on facet name
		// (trying to get proper property uri instead of field name)
		String scope = facet.getName();
		if (fieldToPropertyMap.containsKey(facet.getName())) {
			scope = fieldToPropertyMap.get(facet.getName());
		}
		JenaPoKHelper hlpr = new JenaPoKHelper(AnnotationFactory.createAndLinkAnnotation(facetQuery));
		hlpr.createResStat(facetQuery.getUri(), SolrSearcher.HAS_SCOPE, scope);
		hlpr.commit();
		// ensure expression are expressed with quotes
		if(facetFilter.contains(" ") && !facetFilter.contains("\"")){
			facetFilter =  "\""+facetFilter+"\"";
		}
		// set filter
		facetQuery.setRequest(facetFilter);

		// combine original query and filter in one ComposedQuery
		ComposedQuery cQuery = ResourceFactory.createResource(SolrComponent.IDREF, cQueryIdRes, ComposedQuery.class);
		cQuery.setOperator(Operator.AND);
		cQuery.getQuery().add(query);
		cQuery.getQuery().add(facetQuery);

		// add facet information on the query
		hlpr = new JenaPoKHelper(AnnotationFactory.createAndLinkAnnotation(cQuery));
		hlpr.createLitStat(cQuery.getUri(), WebLabRetrieval.HAS_NUMBER_OF_RESULTS, Integer.toString(nbRes));
		hlpr.createLitStat(cQuery.getUri(), WebLabRetrieval.HAS_QUERY_OFFSET, Integer.toString(offset));
		hlpr.createLitStat(cQuery.getUri(), WebLabRetrieval.IS_EXPRESSED_WITH, this.getClass().getSimpleName());
		// if (fieldToPropertyMap.containsKey(facet.getName())) {
		// hlpr.createResStat(cQuery.getUri(), WebLabQueryParser.HAS_SCOPE,
		// fieldToPropertyMap.get(facet.getName()));
		// } else {
		// hlpr.createResStat(cQuery.getUri(), WebLabQueryParser.HAS_SCOPE,
		// facet.getName());
		// }
		hlpr.createLitStat(cQuery.getUri(), RDFS.LABEL, facetFilter);
		hlpr.createLitStat(cQuery.getUri(), WebLabRetrieval.HAS_DESCRIPTION, facetFilter);
		hlpr.createResStat(cQuery.getUri(), WebLabRetrieval.IS_LINKED_TO, query.getUri());
		hlpr.commit();
		return cQuery;
	}

	private void checkArgs(ProcessArgs args) throws InvalidParameterException {
		if (args == null) {
			throw new InvalidParameterException("Input args for [" + this.getClass().getSimpleName() + "] cannot be null.");
		}
		// TODO finish it...
	}

	public SolrConfig getConf() {
		return conf;
	}

	public void setConf(SolrConfig conf) {
		this.conf = conf;
	}

	public Set<Field> getFacetFields() {
		return facetFields;
	}

	public void setFacetFields(Set<Field> facetFields) {
		this.facetFields = facetFields;
	}
}
