package org.ow2.weblab.service.transcript.sphinx;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.util.Map;

import javax.annotation.PostConstruct;
import javax.jws.WebService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.content.api.ContentManager;
import org.ow2.weblab.core.extended.properties.PropertiesLoader;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.Resource;
import org.ow2.weblab.core.services.AccessDeniedException;
import org.ow2.weblab.core.services.Analyser;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InsufficientResourcesException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.ow2.weblab.core.services.ServiceNotConfiguredException;
import org.ow2.weblab.core.services.UnexpectedException;
import org.ow2.weblab.core.services.UnsupportedRequestException;
import org.ow2.weblab.core.services.analyser.ProcessArgs;
import org.ow2.weblab.core.services.analyser.ProcessReturn;

@WebService(endpointInterface = "org.ow2.weblab.core.services.Analyser")
public class SphinxTranscriptorService implements Analyser {

	private static Log LOG = LogFactory.getLog(SphinxTranscriptorService.class);
	private static final String WRITE_TOKEN_INFO = "write-token-info";
	private static final String WRITE_PRONUNCIATION = "write-pronunciation";
	private static final String WRITE_SCORE = "write-score";
	private static final String WRITE_FILLER = "write-filler";
	private final static String PROPERTY_FILE = "sphinx.properties";
	private final static String TARGET_LANG = "generated-language";
	private final static String SPHINX4_XML_FILE = "sphinx4-xml-file";
	protected WebLabTextTranscriptCreator webLabTextTranscriptCreator;

	public void init(String configFile) throws URISyntaxException, IOException {
		/*
		 * loading property file
		 */
		Map<String, String> props = PropertiesLoader.loadProperties(configFile);

		/*
		 * searching for xml file in class path
		 */
		String sphinx4XmlFilePath = props.get(SPHINX4_XML_FILE);
		File sphinxConfigFile = File.createTempFile("sphinx4", ".xml");
		writeStreamToFile(
				this.getClass().getClassLoader()
						.getResourceAsStream(sphinx4XmlFilePath),
				sphinxConfigFile);

		/*
		 * instanciate transcriptor using property file properties
		 */
		SphinxTranscriptor sphinxTranscriptor = new SphinxTranscriptor(
				sphinxConfigFile.getAbsolutePath());
		webLabTextTranscriptCreator = new WebLabTextTranscriptCreator(sphinxTranscriptor, 
													ContentManager.getInstance(),
													Boolean.valueOf(props.get(WRITE_TOKEN_INFO)),
													Boolean.valueOf(props.get(WRITE_FILLER)), 
													Boolean.valueOf(props.get(WRITE_SCORE)), 
													Boolean.valueOf(props.get(WRITE_PRONUNCIATION)), 
													props.get(TARGET_LANG));
		LOG.info("Service inialized with: " + webLabTextTranscriptCreator);
	}

	@PostConstruct
	public void init() throws URISyntaxException, IOException {
		init(PROPERTY_FILE);
	}

	@Override
	public ProcessReturn process(ProcessArgs args)
			throws AccessDeniedException, ContentNotAvailableException,
			InsufficientResourcesException, InvalidParameterException,
			ServiceNotConfiguredException, UnexpectedException,
			UnsupportedRequestException {
		checkArgs(args);
		Document document = (Document) args.getResource();

		/*
		 * calling WLTranscriptor
		 */
		webLabTextTranscriptCreator.transcriptDocument(document);

		ProcessReturn processReturn = new ProcessReturn();
		processReturn.setResource(document);
		return processReturn;
	}

	/**
	 * @param processArgs
	 *            The processArgs; i.e. a usageContext not used and a Resource
	 *            must be a Document
	 * @throws ProcessException
	 *             If processArgs is null; or if resource is null; or if
	 *             resource is neither a ComposedUnit nor a Audio.
	 */
	private void checkArgs(final ProcessArgs processArg)
			throws AccessDeniedException, ContentNotAvailableException,
			InsufficientResourcesException, InvalidParameterException,
			ServiceNotConfiguredException, UnexpectedException,
			UnsupportedRequestException {
		if (processArg == null) {
			throw new InvalidParameterException("ProcessArgs was null.",
					"ProcessArgs was null.");
		}
		Resource res = processArg.getResource();
		if (res == null) {
			throw new InsufficientResourcesException(
					"Resource in ProcessArgs was null.",
					"Resource in ProcessArgs was null.");
		}
		if (!(res instanceof Document)) {
			throw new InvalidParameterException(
					"Resource in ProcessArgs was not an instance of Document but of ["
							+ res.getClass().getCanonicalName() + "].", "");
		}
	}

	/**
	 * Copy a stream to a file.
	 * 
	 * @param stream
	 *            to copy
	 * @param file
	 *            will contains the stream content
	 * @throws IOException
	 *             error occurs during file copy
	 */
	private void writeStreamToFile(InputStream inputStream, File file)
			throws IOException {
		BufferedReader bufferedReader = new BufferedReader(
				new InputStreamReader(inputStream));
		FileWriter fileWriter = new FileWriter(file);
		String line;
		while ((line = bufferedReader.readLine()) != null) {
			fileWriter.write(line + "\n");
		}
		fileWriter.close();
		bufferedReader.close();
	}
}
