package org.ow2.weblab.service.transcript.sphinx;

import java.io.File;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.DecimalFormat;
import java.util.LinkedList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.weblab.content.api.ContentManager;
import org.ow2.weblab.core.extended.exception.WebLabCheckedException;
import org.ow2.weblab.core.extended.exception.WebLabUncheckedException;
import org.ow2.weblab.core.extended.factory.AnnotationFactory;
import org.ow2.weblab.core.extended.factory.MediaUnitFactory;
import org.ow2.weblab.core.extended.factory.SegmentFactory;
import org.ow2.weblab.core.extended.ontologies.RDF;
import org.ow2.weblab.core.extended.ontologies.RDFS;
import org.ow2.weblab.core.extended.ontologies.WebLabProcessing;
import org.ow2.weblab.core.extended.util.ResourceUtil;
import org.ow2.weblab.core.helper.PoKHelper;
import org.ow2.weblab.core.helper.impl.JenaPoKHelper;
import org.ow2.weblab.core.model.Annotation;
import org.ow2.weblab.core.model.Audio;
import org.ow2.weblab.core.model.Document;
import org.ow2.weblab.core.model.LinearSegment;
import org.ow2.weblab.core.model.TemporalSegment;
import org.ow2.weblab.core.model.Text;
import org.ow2.weblab.core.model.processing.WProcessingAnnotator;
import org.ow2.weblab.core.services.ContentNotAvailableException;
import org.ow2.weblab.core.services.InvalidParameterException;
import org.purl.dc.elements.DublinCoreAnnotator;

import edu.cmu.sphinx.decoder.search.Token;
import edu.cmu.sphinx.frontend.Data;
import edu.cmu.sphinx.frontend.FloatData;
import edu.cmu.sphinx.linguist.WordSearchState;
import edu.cmu.sphinx.linguist.acoustic.Unit;
import edu.cmu.sphinx.linguist.dictionary.Pronunciation;
import edu.cmu.sphinx.linguist.dictionary.Word;

public class WebLabTextTranscriptCreator {

	private static Log LOG = LogFactory.getLog(WebLabTextTranscriptCreator.class);
	
	/**
	 * URI used in isProducedBy property
	 */
	//TODO make it flexible through properties
	private static final String SPHINX_SERVICE_URI = "http://weblab.ow2.org/services/speech2text#sphinx";
	
	private static final String SPHINX_URI = "http://www.speech.cs.cmu.edu/sphinx/ontology#";
	private static final String SPHINX_URI_SPELLING = SPHINX_URI + "spelling";
	private static final String SPHINX_URI_SCORE = SPHINX_URI + "score";
	private static final String SPHINX_URI_PRONUNCIATION = SPHINX_URI + "pronunciation";
	private static final String SPHINX_URI_INSTANCE = SPHINX_URI + "instance";
	private static final String SPHINX_URI_TOKEN_CLASS = SPHINX_URI + "Token";

	private static final DecimalFormat scoreFmt = new DecimalFormat("0.00E00");
	protected SphinxTranscriptor sphinxTranscriptor;
	protected boolean writeTokenInfo;
	protected boolean writeFiller;
	protected boolean writeScore;
	protected boolean writePronunciation;
	protected String targetLang;
	protected ContentManager contentManager;
	
	/**
	 * Generates all text transcript from all audio media unit contained in a
	 * WebLab document using a specific sphinxTranscriptor
	 * 
	 * @param sphinxTranscriptor
	 *            model used to transcript
	 * @param contentManager
	 *            content manager to be used
	 * @param writeTokenInfo
	 * 			  true if your want to add extra information on each token     
	 * @param writeFiller
	 *            true if you want filler words
	 * @param writeScore
	 *            true if you want to write score
	 * @param writePronunciation
	 *            true if you want to write pronunciation
	 * @param targetLang
	 *            target language
	 */
	public WebLabTextTranscriptCreator(SphinxTranscriptor sphinxTranscriptor, ContentManager contentManager, boolean writeTokenInfo, boolean writeFiller, boolean writeScore,
			boolean writePronunciation, String targetLang) {
		super();
		this.sphinxTranscriptor = sphinxTranscriptor;
		this.contentManager = contentManager;
		this.writeTokenInfo = writeTokenInfo;
		this.writeFiller = writeFiller;
		this.writeScore = writeScore;
		this.writePronunciation = writePronunciation;
		this.targetLang = targetLang;
	}

	public void transcriptDocument(Document parent) throws InvalidParameterException, ContentNotAvailableException {
		/*
		 * gets all audio units.
		 */
		List<Audio> audioList = ResourceUtil.getSelectedSubResources(parent, Audio.class);
		/*
		 * check if there is at least one audio unit
		 */
		if (audioList.isEmpty()) {
			LOG.warn("No audio found in Document " + parent.getUri());
			throw new InvalidParameterException("No audio media unit found, escaping transcription.", "No audio media unit found in document "
					+ parent.getUri() + ", escaping transcription.");
		}

		/*
		 * for each audio content in the document
		 */
		for (Audio audio : audioList) {
			/*
			 * get the audio normalised content
			 */

			File normalisedAudioFile;
			try {
				normalisedAudioFile = contentManager.readNormalisedContent(audio);
			} catch (WebLabCheckedException e) {
				throw new ContentNotAvailableException("Error during content retrieving on " + parent.getUri(), e.getMessage(), e);
			}

			/*
			 * creates transcripted Text media unit and links it to current
			 * Audio
			 */
			Text text = MediaUnitFactory.createAndLinkMediaUnit(parent, Text.class);
			WProcessingAnnotator textProcessingAnnotator = new WProcessingAnnotator(text);
			DublinCoreAnnotator dublinCoreAnnotator = new DublinCoreAnnotator(text);
			dublinCoreAnnotator.writeLanguage(this.targetLang);
			
			try {
				textProcessingAnnotator.writeTranscriptOf(new URI(audio.getUri()));
				textProcessingAnnotator.writeProducedBy(new URI(SPHINX_SERVICE_URI));
			} catch (URISyntaxException e) {
				/*
				 * should never occurs
				 */
				throw new WebLabUncheckedException("Auto generated Audio mediaUnit URI is invalid: " + audio.getUri(), e);
			}

			/*
			 * creates annotation unit
			 */
			Annotation textAnnotation = AnnotationFactory.createAndLinkAnnotation(text);
			Annotation audioAnnotation = AnnotationFactory.createAndLinkAnnotation(audio);
			
			/*
			 * for each transcripted text section for this audio file
			 */
			for (Token token : sphinxTranscriptor.transcript(normalisedAudioFile)) {
				generatedTranscriptedText(token, text, audio, textAnnotation, audioAnnotation);
			}
		}
	}

	/**
	 * 
	 * Creates aligned segments from Token, Text and Audio. inspired from Sphinx
	 * code in Result.getTimedWordPath
	 * 
	 * @param token
	 *            Sphinx Token
	 * @param text
	 *            Text media unit transcripted
	 * @param audio
	 *            Audio source media unit
	 * @param audioAnnotation
	 *            Annotation on Audio unit
	 * @param textAnnotation
	 *            Annotation on Text unit
	 */
	protected void generatedTranscriptedText(Token curToken, Text text, Audio audio, Annotation textAnnotation, Annotation audioAnnotation) {

		StringBuffer transcriptedBuffer = new StringBuffer();
		/*
		 * get to the first emitting token
		 */
		while (curToken != null && curToken.getSearchState() != null && !curToken.isEmitting()) {
			curToken = curToken.getPredecessor();
		}

		/*
		 * creates a list of token in the right order (to generate directly the
		 * text segment in the right order
		 */
		LinkedList<Token> orderredToken = new LinkedList<Token>();
		while (curToken != null) {
			orderredToken.addFirst(curToken);
			curToken = curToken.getPredecessor();
		}

		/*
		 * for each token
		 */
		Data lastWordFirstFeature = null, lastFeature = null;

		for (Token token : orderredToken) {
			if (lastWordFirstFeature == null) {
				lastWordFirstFeature = token.getData();
				lastFeature = lastWordFirstFeature;
			}
			if (token.isWord()) {
				/*
				 * token is a word, going to create two segments one temporal on
				 * audio and one linear on text
				 */

				Word word = token.getWord();
				if (this.writeFiller || !word.isFiller()) {
					addWord(transcriptedBuffer, textAnnotation, audioAnnotation, text, audio, token, (FloatData) lastFeature, (FloatData) lastWordFirstFeature);
				}
				lastWordFirstFeature = lastFeature;
			}
			Data feature = token.getData();
			if (feature != null) {
				lastFeature = feature;
			}
		}

		String transcriptedString = transcriptedBuffer.toString().trim();
		/*
		 * nothing to write
		 */
		if (transcriptedString.length() > 0) {
			if (text.getContent() != null && text.getContent().length() > 0) {
				text.setContent(text.getContent() + transcriptedString + "\n");
			} else {
				text.setContent(transcriptedString + "\n");
			}
			// if we do not write any extra info, then audio Annotation should be removed
			if(!(this.writeFiller && this.writePronunciation && this.writeScore && this.writeTokenInfo))
				audio.getAnnotation().remove(audioAnnotation);
		} else {
			/*
			 * removing unused annotations
			 */
			text.getAnnotation().remove(textAnnotation);
			audio.getAnnotation().remove(audioAnnotation);
		}

	}

	/**
	 * Generates the two segments with all configured annotations
	 * 
	 * @param textAnnotation
	 *            Annotation where to add text segment metadata
	 * @param audioAnnotation
	 *            Annotation where to add audio segment metadata
	 * @param text
	 *            Text unit to add linear segment to
	 * @param audio
	 *            Audio unit to add temporal segment to
	 * @param token
	 *            the Sphinx recognnized token
	 * @param startFeature
	 *            Sphinx started feature
	 * @param endFeature
	 *            Sphinx ending feature
	 */
	protected void addWord(StringBuffer sb, Annotation textAnnotation, Annotation audioAnnotation, Text text, Audio audio, Token token, FloatData startFeature,
			FloatData endFeature) {
		Word word = token.getWord();
		/*
		 * temporal segment
		 */
		int startTime = startFeature == null ? -1 : (int) ((float) startFeature.getFirstSampleNumber() * 1000 / startFeature.getSampleRate());
		int endTime = endFeature == null ? -1 : (int) ((float) endFeature.getFirstSampleNumber() * 1000 / endFeature.getSampleRate());
		TemporalSegment temporalSegment = null;
		if (endTime != -1 && startTime != -1) {
			temporalSegment = SegmentFactory.createAndLinkSegment(audio, TemporalSegment.class);
			temporalSegment.setStart(startTime);
			temporalSegment.setEnd(endTime);

			/*
			 * linear segment
			 */
			int startPosition = sb.length();
			if (text.getContent() != null) {
				startPosition += text.getContent().length();
			}

			int endPosition = startPosition + word.getSpelling().length();
			LinearSegment linearSegment = SegmentFactory.createAndLinkLinearSegment(text, startPosition, endPosition);

			/*
			 * add the word to the buffer
			 */
			sb.append(word.getSpelling() + " ");

			/*
			 * segments alignment and annotation
			 */
			try {

				URI linearSegmentUri = new URI(linearSegment.getUri());
				URI temporalSegmentUri = new URI(temporalSegment.getUri());

				/*
				 * generates pokH on the audio annotation
				 */
				PoKHelper audioPoKHelper = new JenaPoKHelper(audioAnnotation);

				audioPoKHelper.setAutoCommitMode(false);

				/*
				 * generates unique temp uri for the token item
				 */
				String tokenUri = SPHINX_URI_INSTANCE + System.nanoTime();

				/*
				 * get meta data values
				 */
				String score = scoreFmt.format(token.getScore());
				String pronuniation = getPronuniation(token);
				String spelling = word.getSpelling();

				if(this.writeTokenInfo){
					/*
					 * links segment to token
					 */
					audioPoKHelper.createResStat(temporalSegment.getUri(), WebLabProcessing.IS_DELIMITER_OF_TRANSCRIPT_UNIT, text.getUri());
	
					/*
					 * writes token type
					 */
					audioPoKHelper.createResStat(tokenUri, RDF.TYPE, SPHINX_URI_TOKEN_CLASS);
	
					/*
					 * writes token spelling
					 */
					audioPoKHelper.createLitStat(tokenUri, SPHINX_URI_SPELLING, spelling);
	
					/*
					 * writes token label
					 */
					audioPoKHelper.createLitStat(tokenUri, RDFS.LABEL, spelling + " [" + pronuniation + " ] @ (" + score + ")");
				}
				if (this.writeScore) {
					/*
					 * writes token score
					 */
					audioPoKHelper.createLitStat(tokenUri, SPHINX_URI_SCORE, score);
				}
				if (this.writePronunciation) {
					/*
					 * writes token pronunciation
					 */
					audioPoKHelper.createLitStat(tokenUri, SPHINX_URI_PRONUNCIATION, pronuniation);
				}

				audioPoKHelper.commit();

				WProcessingAnnotator textAnnotator = new WProcessingAnnotator(linearSegmentUri, textAnnotation);
				textAnnotator.writeGeneratedFrom(temporalSegmentUri);

			} catch (URISyntaxException e) {
				/*
				 * should never occurs
				 */
				throw new WebLabUncheckedException("Auto generated Annotation URI is invalid.", e);
			}
		}

	}

	/**
	 * Extract a readable pronunciation representation for a token Inspired from
	 * Token.getWordPath() method
	 * 
	 * @param token
	 *            Token to extract pronunciation
	 * @return A String representing the token pronunciation
	 */
	private String getPronuniation(Token token) {
		StringBuffer sb = new StringBuffer();
		if (token.isWord()) {
			Pronunciation pron = ((WordSearchState) token.getSearchState()).getPronunciation();
			Unit[] u = pron.getUnits();
			for (int i = u.length - 1; i >= 0; i--) {
				if (i < u.length - 1)
					sb.insert(0, ',');
				sb.insert(0, u[i].getName());
			}
		}
		return sb.toString();
	}

	public String toString() {
		return "WebLabTextTranscriptor writing (filler, score, pronunciation): " + writeFiller + ", " + writeScore + ", " + writePronunciation + " using "
				+ sphinxTranscriptor + " as transcriptor and " + contentManager + " as content manager.";
	}
}