package org.ow2.weblab.service.normaliser.tika;

import java.text.MessageFormat;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.apache.commons.logging.LogFactory;

/**
 * A utility class that enable to access the resource bundle and to use MessageFormat parsing for parametrised messages.
 * 
 * @author WebLab IPCC Team
 * @date 2011-06-16
 */
public abstract class Messages {


	private static final ResourceBundle RESOURCE_BUNDLE = ResourceBundle.getBundle(Messages.class.getCanonicalName());


	private static final String MSG_I18N_NOT_WORKING = "The internationalisation of Messages is not working...";


	private Messages() {
		// Should not be instantiated.
	}


	/**
	 * Lookup in the ResourceBundle for the key in parameter.
	 * 
	 * @param key
	 *            The key in the resource bundle.
	 * 
	 * @return The value in the resource file.
	 */
	public static String getString(final String key) {
		try {
			return Messages.RESOURCE_BUNDLE.getString(key);
		} catch (final MissingResourceException mre) {
			try {
				LogFactory.getLog(Messages.class).warn(Messages.getString(Constants.KEY_WARN_MSG_NOT_FOUND_1, key), mre);
			} catch (final MissingResourceException mre2) {
				LogFactory.getLog(Messages.class).warn(Messages.MSG_I18N_NOT_WORKING);
			}
			return '!' + key + '!';
		}
	}


	/**
	 * Lookup in the ResourceBundle for the key in parameter. The parameters in ellipse are used through the {@link MessageFormat#format(String, Object...)}
	 * method.
	 * 
	 * @param key
	 *            The key of the message.
	 * @param params
	 *            The array used by the inner message formatter class to replace informations in the message.
	 * @return The value associated to the key in the property file, or !key!. Using the array in parameter to replace some values.
	 */
	public static String getString(final String key, final Object... params) {
		final String pattern;
		try {
			pattern = Messages.RESOURCE_BUNDLE.getString(key);
		} catch (final MissingResourceException mre) {
			if (Constants.KEY_WARN_MSG_NOT_FOUND_1.equals(key)) {
				LogFactory.getLog(Messages.class).warn(Messages.MSG_I18N_NOT_WORKING);
			} else {
				try {
					LogFactory.getLog(Messages.class).warn(Messages.getString(Constants.KEY_WARN_MSG_NOT_FOUND_1, key), mre);
				} catch (final MissingResourceException mre2) {
					LogFactory.getLog(Messages.class).warn(Messages.MSG_I18N_NOT_WORKING);
				}
			}
			return '!' + key + '!';
		}
		return MessageFormat.format(pattern, params);
	}
}
