/**
 * WEBLAB: Service oriented integration platform for media mining and intelligence applications
 * 
 * Copyright (C) 2004 - 2011 Cassidian, an EADS company
 * 
 * This library is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 51 Franklin Street, Fifth
 * Floor, Boston, MA 02110-1301 USA
 */
package org.ow2.weblab.service.normaliser.tika;

import java.net.URI;

import org.ow2.weblab.service.normaliser.tika.handlers.SimpleTextContentHandler;
import org.ow2.weblab.service.normaliser.tika.handlers.WebLabHandlerDecorator;

/**
 * This class is just an holder for the configuration of the Tika Service.
 * 
 * 
 * @author WebLal IPCC Team
 * @date 2011-12-01
 */
public class TikaConfiguration {

	/**
	 * The default value for unmappedPropertiesBaseUri.
	 */
	public final static String UNMAPPED_PROPERTIES_BASE_URI = "http://tika.apache.org/property#";

	/**
	 * The default value for unmappedPropertiesPrefix.
	 */
	public final static String UNMAPPED_PROPERTIES_PREFIX = "tika";

	/**
	 * The default weblabHandlerDecorator
	 */
	public final static Class<? extends WebLabHandlerDecorator> WEBLAB_HANDLER_DECORATOR = SimpleTextContentHandler.class;

	/**
	 * Path to tika XML configuration file. If null, default TikaConfig is used.
	 */
	private String pathToXmlConfigurationFile;

	/**
	 * Whether to annotate the document with metadata guessed by Tika.
	 */
	private boolean addMetadata;

	/**
	 * Whether or not to annotate properties from Tika that are not mapped to WebLab Processing, Dublin Core or Dublin Core Terms ontologies.
	 */
	private boolean addUnmappedProperties;

	/**
	 * Whether to annotate the language at the document level, i.e. with a concatenated String of the whole Text content.
	 */
	private boolean annotateDocumentWithLang;

	/**
	 * The default language to be used, if Tika cannot guess it. If empty of null, the MediaUnit will simply not be annotate.
	 */
	private String defaultLang;

	/**
	 * Whether or not to generate an HTML pivot representation of the content. It will be stored using the ContentManager.
	 */
	private boolean generateHtml;

	/**
	 * Sometimes the document already contains annotations coming from previous services (creator, created, etc.). This value defines whether to override
	 * existing metadata or not to annotate with the Tika value if the annotation already exists.
	 */
	private boolean overrideMetadata;

	/**
	 * If the content is a temporary one (the local copy after a transfer through FTP or WebDAV for instance), whether to remove it after a successful process
	 * or not.
	 */
	private boolean removeTempContent;

	/**
	 * The service URI to be used to annotate the created annotations. If null, nothing will be done.
	 */
	private URI serviceUri;

	/**
	 * The baseURI to be used for annotating properties from Tika that are not mapped to WebLab Processing, Dublin Core or Dublin Core Terms ontologies.
	 * 
	 * If <code>null</code> or empty the default value is {@value UNMAPPED_PROPERTIES_BASE_URI}.
	 */
	private String unmappedPropertiesBaseUri;

	/**
	 * The xml namespace prefix to be used for annotating properties from Tika that are not mapped to WebLab Processing, Dublin Core or Dublin Core Terms
	 * ontologies.
	 * 
	 * If <code>null</code> or empty the default value is {@value UNMAPPED_PROPERTIES_PREFIX}.
	 */
	private String unmappedPropertiesPrefix;

	/**
	 * The class of the WebLab Handler Decorator that will be instantiated to convert the HTML into a WebLab document.
	 * 
	 * If <code>null</code> the default value is {@value WEBLAB_HANDLER_DECORATOR}
	 */
	private Class<? extends WebLabHandlerDecorator> webLabHandlerDecoratorClass;

	/**
	 * The constructor
	 */
	public TikaConfiguration() {
		super();
	}

	/**
	 * 
	 * @return the path to xml tika config file
	 */
	public String getPathToXmlConfigurationFile() {
		return pathToXmlConfigurationFile;
	}

	/**
	 * 
	 * @param pathToXmlConfigurationFile
	 *            the path to xml tika config file
	 */
	public void setPathToXmlConfigurationFile(String pathToXmlConfigurationFile) {
		this.pathToXmlConfigurationFile = pathToXmlConfigurationFile;
	}

	/**
	 * 
	 * @param serviceUri
	 *            the service URI (used to annotated the annotations)
	 */
	public void setServiceUri(URI serviceUri) {
		this.serviceUri = serviceUri;
	}

	/**
	 * @return the defaultLang
	 */
	public String getDefaultLang() {
		return this.defaultLang;
	}

	/**
	 * @return the serviceUri
	 */
	public String getServiceUri() {
		if (this.serviceUri == null) {
			return null;
		}
		return this.serviceUri.toString();
	}

	/**
	 * @return the unmappedPropertiesBaseUri
	 */
	public String getUnmappedPropertiesBaseUri() {
		if ((this.unmappedPropertiesBaseUri == null) || this.unmappedPropertiesBaseUri.isEmpty()) {
			return TikaConfiguration.UNMAPPED_PROPERTIES_BASE_URI;
		}
		return this.unmappedPropertiesBaseUri;
	}

	/**
	 * @return the unmappedPropertiesPrefix
	 */
	public String getUnmappedPropertiesPrefix() {
		if ((this.unmappedPropertiesPrefix == null) || this.unmappedPropertiesPrefix.isEmpty()) {
			return TikaConfiguration.UNMAPPED_PROPERTIES_PREFIX;
		}
		return this.unmappedPropertiesPrefix;
	}

	/**
	 * @return the webLabHandlerDecoratorClass
	 */
	public Class<? extends WebLabHandlerDecorator> getWebLabHandlerDecoratorClass() {
		if (this.webLabHandlerDecoratorClass == null) {
			return TikaConfiguration.WEBLAB_HANDLER_DECORATOR;
		}
		return this.webLabHandlerDecoratorClass;
	}

	/**
	 * @return the addMetadata
	 */
	public boolean isAddMetadata() {
		return this.addMetadata;
	}

	/**
	 * @return the addUnmappedProperties
	 */
	public boolean isAddUnmappedProperties() {
		return this.addUnmappedProperties;
	}

	/**
	 * @return the annotateDocumentWithLang
	 */
	public boolean isAnnotateDocumentWithLang() {
		return this.annotateDocumentWithLang;
	}

	/**
	 * @return the generateHtml
	 */
	public boolean isGenerateHtml() {
		return this.generateHtml;
	}

	/**
	 * @return the overrideMetadata
	 */
	public boolean isOverrideMetadata() {
		return this.overrideMetadata;
	}

	/**
	 * @return the removeTempContent
	 */
	public boolean isRemoveTempContent() {
		return this.removeTempContent;
	}

	/**
	 * @param addMetadata
	 *            the addMetadata to set
	 */
	public void setAddMetadata(final boolean addMetadata) {
		this.addMetadata = addMetadata;
	}

	/**
	 * @param addUnmappedProperties
	 *            the addUnmappedProperties to set
	 */
	public void setAddUnmappedProperties(final boolean addUnmappedProperties) {
		this.addUnmappedProperties = addUnmappedProperties;
	}

	/**
	 * @param annotateDocumentWithLang
	 *            the annotateDocumentWithLang to set
	 */
	public void setAnnotateDocumentWithLang(final boolean annotateDocumentWithLang) {
		this.annotateDocumentWithLang = annotateDocumentWithLang;
	}

	/**
	 * @param defaultLang
	 *            the defaultLang to set
	 */
	public void setDefaultLang(final String defaultLang) {
		this.defaultLang = defaultLang;
	}

	/**
	 * @param generateHtml
	 *            the generateHtml to set
	 */
	public void setGenerateHtml(final boolean generateHtml) {
		this.generateHtml = generateHtml;
	}

	/**
	 * @param overrideMetadata
	 *            the overrideMetadata to set
	 */
	public void setOverrideMetadata(final boolean overrideMetadata) {
		this.overrideMetadata = overrideMetadata;
	}

	/**
	 * @param removeTempContent
	 *            the removeTempContent to set
	 */
	public void setRemoveTempContent(final boolean removeTempContent) {
		this.removeTempContent = removeTempContent;
	}

	/**
	 * @param serviceUri
	 *            the serviceUri to set
	 */
	public void setServiceUri(final String serviceUri) {
		if ((serviceUri == null) || serviceUri.isEmpty()) {
			this.serviceUri = null;
		} else {
			this.serviceUri = URI.create(serviceUri);
		}

	}

	/**
	 * @param unmappedPropertiesBaseUri
	 *            the unmappedPropertiesBaseUri to set
	 */
	public void setUnmappedPropertiesBaseUri(final String unmappedPropertiesBaseUri) {
		this.unmappedPropertiesBaseUri = unmappedPropertiesBaseUri;
	}

	/**
	 * @param unmappedPropertiesPrefix
	 *            the unmappedPropertiesPrefix to set
	 */
	public void setUnmappedPropertiesPrefix(final String unmappedPropertiesPrefix) {
		this.unmappedPropertiesPrefix = unmappedPropertiesPrefix;
	}

	/**
	 * @param webLabHandlerDecoratorClass
	 *            the webLabHandlerDecoratorClass to set
	 */
	public void setWebLabHandlerDecoratorClass(final Class<? extends WebLabHandlerDecorator> webLabHandlerDecoratorClass) {
		this.webLabHandlerDecoratorClass = webLabHandlerDecoratorClass;
	}

}
