package org.ow2.weblab.service.normaliser.tika.metadatawriter;

import java.net.URI;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tika.metadata.Metadata;
import org.apache.tika.metadata.Property;
import org.joda.time.DateTime;
import org.joda.time.format.ISODateTimeFormat;
import org.ow2.weblab.core.helper.PoKHelper;

/**
 * This class aims to provide a mapping between a Tika metadata and an
 * annotation represented by an URI. Currently only Date and URI are used,
 * others are mapped as text properties. See
 * {@link org.apache.tika.metadata.Property.ValueType Property.ValueType} for
 * more info.
 * 
 * @author Frédéric Praca
 * @see org.ow2.weblab.core.helper.PoKHelper
 * @see org.apache.tika.metadata.Metadata
 * @see java.net.URI
 * @see org.apache.tika.metadata.Property
 */
public class MappingDescription {

	private URI annotation_URI;
	private Property propertyLinked;

	/**
	 * The logger to be used inside this class.
	 */
	protected final Log logger;

	/**
	 * Constructor that take a {@link Property XMP Property} as argument and the
	 * {@link URI URI} that should map it. For Tika metadata not represented
	 * with a Property object, it should be represented as Text property
	 * 
	 * @param property
	 *            the Tika property to map
	 * @param uri
	 *            The URI representing the mapping
	 */
	public MappingDescription(final Property property, final URI uri) {
		logger = LogFactory.getLog(this.getClass());
		annotation_URI = uri;
		propertyLinked = property;
	}

	/**
	 * This method is a delegate for
	 * {@linkplain MetadataWriter#write(Metadata, PoKHelper, URI)}.
	 * 
	 * @param metadataToWrite
	 *            Metadata set to write
	 * @param pokHelperToWriteInto
	 *            PokHelper to write into
	 * @param documentUri
	 *            The document URI that will be used for reference
	 */
	public void write(final Metadata metadataToWrite,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {

		// Retrieve the value associated to the linked property
		switch (propertyLinked.getValueType()) {
		case DATE:
			writeDate(metadataToWrite, pokHelperToWriteInto, documentUri);
			break;

		case URI:
			writeURI(metadataToWrite, pokHelperToWriteInto, documentUri);
			break;

		default:
			writeText(metadataToWrite, pokHelperToWriteInto, documentUri);
			break;
		}
	}

	/**
	 * This method defines how to write a Text metadata. It's called from
	 * {@link #write(Metadata, PoKHelper, URI) write} and simply writes the text
	 * as is. It should only be overridden in subclasses that needs to modify or
	 * add some text value to the property.
	 * 
	 * @param metadataToWrite
	 *            Metadata set to write
	 * @param pokHelperToWriteInto
	 *            PokHelper to write into
	 * @param documentUri
	 *            The document URI that will be used for reference
	 */
	protected void writeText(final Metadata metadataToWrite,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {
		String value_l = metadataToWrite.get(propertyLinked);
		if (!value_l.isEmpty()) {
			pokHelperToWriteInto.createLitStat(documentUri.toString(),
					annotation_URI.toString(), value_l);
		} else {
			logger.warn("Property " + propertyLinked.getName() + " was empty");
		}

	}

	/**
	 * This method defines how to write an URI metadata. It's called from
	 * {@link #write(Metadata, PoKHelper, URI) write} and writes the URI using
	 * {@link PoKHelper#createResStat(String, String, String) createResStat} of
	 * PokHelper. It should only be overridden in subclasses that needs to
	 * modify the URI of the property.
	 * 
	 * @param metadataToWrite
	 *            Metadata set to write
	 * @param pokHelperToWriteInto
	 *            PokHelper to write into
	 * @param documentUri
	 *            The document URI that will be used for reference
	 */
	protected void writeURI(final Metadata metadataToWrite,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {
		pokHelperToWriteInto.createResStat(documentUri.toString(),
				annotation_URI.toString(), metadataToWrite.get(propertyLinked));
	}

	/**
	 * This method defines how to write a Date metadata. It's called from
	 * {@link #write(Metadata, PoKHelper, URI) write} and writes the Date using
	 * Joda Time. It should only be overridden in subclasses that needs to
	 * modify, for example, the Date format of the property.
	 * 
	 * @param metadataToWrite
	 *            Metadata set to write
	 * @param pokHelperToWriteInto
	 *            PokHelper to write into
	 * @param documentUri
	 *            The document URI that will be used for reference
	 */
	protected void writeDate(final Metadata metadataToWrite,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {
		Date localDate_l = metadataToWrite.getDate(propertyLinked);
		if (localDate_l == null) {
			logger.warn("Date property " + propertyLinked.getName()
					+ " was null, setting to current date");
		} else {
			DateTime dt = new DateTime(localDate_l);
			pokHelperToWriteInto.createLitStat(documentUri.toString(),
					annotation_URI.toString(), ISODateTimeFormat.dateTime()
							.print(dt));
		}
	}

	/**
	 * Protected getter for destination URI
	 * 
	 * @return the URI to map to
	 */
	protected URI getAnnotation_URI() {
		return annotation_URI;
	}

	/**
	 * Protected getter for property to map
	 * 
	 * @return the Property to map from
	 */
	protected Property getPropertyLinked() {
		return propertyLinked;
	}
}
