package org.ow2.weblab.service.normaliser.tika.metadatawriter;

import java.net.URI;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.tika.metadata.Metadata;
import org.ow2.weblab.core.helper.PoKHelper;

/**
 * This class is the base class for MetadataWriters. It contains the mapping
 * between metadata as found by Tika to Weblab ontology (or any ontology you'd
 * like) and write these metadata in the Weblab resource according to the
 * PokHelper given. As this class does not contain any mapping, a subclass must
 * be created. This subclass have to provide an implementation for
 * {@link #init() init} method which basically makes calls to
 * {@link #addMapping(String, MappingDescription) addMapping} method. This
 * method takes {@link MappingDescription MappingDescription} (or one of its
 * subclasses) instances.
 * 
 * @author Frédéric Praca
 * 
 * @see org.ow2.weblab.core.helper.PoKHelper
 * @see org.apache.tika.metadata.Metadata
 * @see org.ow2.weblab.service.normaliser.tika.metadatawriter.MappingDescription
 */
public abstract class MetadataWriter {

	/**
	 * The logger to be used inside this class.
	 */
	protected final Log logger;

	/**
	 * Map to store the mapping between Metadata name and MappingDescription
	 * objects
	 */
	private Map<String, MappingDescription> mappingMap = new HashMap<String, MappingDescription>();

	/**
	 * Default constructor used by subclasses
	 */
	public MetadataWriter() {
		super();
		logger = LogFactory.getLog(this.getClass());
		init();
	}

	/**
	 * This method allows to write the Metadata created by Tika inside the
	 * correct part of the document pointed out by a {@link PoKHelper PokHelper}
	 * instance. Each annotation is written to reference the URI of the document
	 * provided.
	 * 
	 * @param metadataToWrite
	 *            The {@link org.apache.tika.metadata.Metadata Metadata} object
	 *            to map and write
	 * @param pokHelperToWriteInto
	 *            The {@link PoKHelper PokHelper} used to write into the
	 *            document
	 * @param documentUri
	 *            The {@link java.net.URI URI} of the document that should be
	 *            used for reference
	 */
	public void write(final Metadata metadataToWrite,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {

		for (final String name : metadataToWrite.names()) {
			// Select the correct MappingDescription to write
			// the data correctly
			MappingDescription desc_l = mappingMap.get(name);
			if (desc_l != null) {
				logger.trace("Writing " + name + " property into " + desc_l.getAnnotation_URI().toASCIIString());
				desc_l.write(metadataToWrite, pokHelperToWriteInto, documentUri);
			} else {
				writeUnmmappedMetadata(metadataToWrite, name, pokHelperToWriteInto, documentUri);
			}
		}
	}

	/**
	 * Method used by subclass in their {@link #init() init} method to add a mapping
	 * @param propertyToMap the Tika property name to map
	 * @param toDescription the description to use for mapping. Can be instance of {@link MappingDescription MappingDescription} or one of its subclass.
	 */
	protected void addMapping(String propertyToMap,
			MappingDescription toDescription) {
		mappingMap.put(propertyToMap, toDescription);
	}

	/**
	 * Method to implement in subclass. Should basically be a set of calls to {@link #addMapping(String, MappingDescription) addMapping}
	 */
	protected abstract void init();
	
	/**
	 * This method aims to write a metadata that was not mapped.
	 * By default, this method only logs in debug mode that the property was not mapped.
	 * For a different behaviour, this method should be overridden
	 * @param metadataToWrite
	 *            The {@link org.apache.tika.metadata.Metadata Metadata} object
	 *            to map and write
	 * @param name Name of the metadata which is not mapped
	 *            
	 * @param pokHelperToWriteInto
	 *            The {@link PoKHelper PokHelper} used to write into the
	 *            document
	 * @param documentUri
	 *            The {@link java.net.URI URI} of the document that should be
	 *            used for reference
	 */
	protected void writeUnmmappedMetadata(final Metadata metadataToWrite, final String name,
			final PoKHelper pokHelperToWriteInto, final URI documentUri) {
		logger.debug("Property " + name + " is not currently mapped, value found : " + metadataToWrite.get(name));
	}
}
