/**
 *	This file is part of org.ow2.wildcat.
 *
 *	wildcat is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	wildcat is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 */
package org.ow2.wildcat;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import net.esper.client.UpdateListener;

import org.ow2.wildcat.event.WEvent;
import org.ow2.wildcat.hierarchy.attribute.Attribute;
import org.ow2.wildcat.hierarchy.resource.Resource;

/**
 * Public interface through which users perform push/pull operation on context data.
 */
public interface Context {

	/**
	 * @return the name of the context as registered for remote connection
	 */
	String getName();

	/*
	 * PULL RELATED OPERATIONS
	 */

	/**
	 * Attach Attribute at specified location.
	 *
	 * @param attribute
	 *            path of the Attribute
	 * @param attr
	 *            an Attribute
	 * @throws ContextException if the attribute cannot be attached
	 */
	void attachAttribute(String attribute, Attribute attr) throws ContextException;

	/**
	 * Creates a new attribute.
	 *
	 * @param attribute
	 *            path of the attribute
	 * @param value
	 *            initial value of the attribute
	 * @throws ContextException if the attribute cannot be created
	 */
	void createAttribute(String attribute, Object value) throws ContextException;

	/**
	 * Detach an attribute.
	 * @param attribute path of the attribute
	 * @throws ContextException if the attribute cannot be detached
	 */
	void detachAttribute(String attribute) throws ContextException;

	/**
	 * Creates Resource along a path
	 *
	 * @param resource
	 *            path to create
	 * @throws ContextException if the resource cannot be created
	 */
	void createResource(String resource) throws ContextException;

	/**
	 * Return the resource at specified location.
	 * @param resource path of the Resource
	 * @return the resource at specified location
	 * @throws ContextException if the resource cannot be returned
	 */
	Resource getResource(String resource) throws ContextException;

	/**
	 * Attach Resource at specified location.
	 * @param resource path of the Resource
	 * @param res a Resource
	 * @throws ContextException if the resource cannot be attached
	 */
	void attachResource(String resource, Resource res) throws ContextException;

	/**
	 * Detach a resource.
	 * @param resource path of the Resource
	 * @throws ContextException if the resource cannot be detached
	 */
	void detachResource(String resource) throws ContextException;

	/**
	 * Creates and attach a Symlink.
	 *
	 * @param link attach point
	 * @param target target path of the symbolic link
	 * @throws ContextException if the Symlink cannot be created
	 */
	void createSymlink(String link, String target) throws ContextException;

	/**
	 * Return the attribute at specified location.
	 * @param attribute the path of the attribute
	 * @return an attribute
	 * @throws ContextException if the attribute cannot be retrieved
	 */
	Attribute getAttribute(String attribute) throws ContextException;

	/**
	 * Fetch attribute value.
	 *
	 * @param attribute
	 *            the path of the attribute
	 * @return the value of the attribute, or <b>null</b> if attribute does not exist
	 * @throws ContextException if the attribute cannot be accessed
	 */
	Object getValue(String attribute) throws ContextException;

	/**
	 * Set Attribute's value.
	 *
	 * @param attribute
	 *            path of the Attribute
	 * @param value
	 *            new value of the Attribute
	 * @return previous value of the Attribute
	 * @throws ContextException  if the attribute cannot be accessed
	 */
	Object setValue(String attribute, Object value) throws ContextException;

	/**
	 * List resources and attributes.
	 *
	 * @param resource
	 *            the resource to inspect
	 * @return A set of String, attribute's names preceded with '#'
	 * @throws ContextException if the resource cannot be accessed
	 */
	Set<String> list(String resource) throws ContextException;

	/*
	 * PUSH RELATED OPERATIONS
	 */

	/**
	 * Emit event(s).
	 *
	 * @param events
	 *            one or more event to emit
	 */
	void emitEvent(WEvent... events);

	/**
	 * Periodically creates an AttributeChangedEvent where old and new values
	 * are equal.
	 *
	 * @param attribute
	 *            the location of the attribute to poll
	 * @param duration
	 *            the period
	 * @param unit
	 *            the period unit
	 * @throws ContextException if the attribute poller cannot be created
	 */
	void createPeriodicAttributePoller(String attribute, long duration,
			TimeUnit unit) throws ContextException;

	/**
	 * Remove an attribute poller.
	 * @param attribute the location of the attribute to poll
	 * @throws ContextException if the attribute poller cannot be deleted
	 */
	void removePeriodicAttributePoller(String attribute) throws ContextException;

	/**
	 * Creates and starts a query.
	 *
	 * @param query
	 *            the query
	 * @return a reference to the query
	 */
	Query createQuery(String query);

	/**
	 * Pause Query.
	 *
	 * @param query
	 *            query's reference
	 */
	void pauseQuery(Query query);

	/**
	 * Resume Query.
	 *
	 * @param query
	 *            query's reference
	 */
	void resumeQuery(Query query);

	/**
	 * Detach all listener and destroy query
	 *
	 * @param query
	 *            the reference of the query
	 */
	void destroyQuery(Query query);

	/**
	 * Creates and starts a query and register one or more listener
	 *
	 * @param query
	 *            the query
	 * @param listeners
	 *            0 or more listeners
	 * @return a reference to the query
	 */
	Query registerListeners(String query, UpdateListener... listeners);

	/**
	 * Attach one or more listener to a query
	 *
	 * @param query
	 *            query reference
	 * @param listeners
	 *            one or more listener to attach
	 */
	void registerListeners(Query query, UpdateListener... listeners);

	/**
	 * Removes one or more listeners associated with a Query
	 *
	 * @param query
	 *            the query from which to detach
	 * @param listeners
	 *            the listeners to detach
	 */
	void removeListeners(Query query, UpdateListener... listeners);

	/*
	 * Distribution related
	 */

	/**
	 * Allow the context for remote connections.
	 * @throws ContextException if this context cannot be exported
	 */
	void export() throws ContextException;

	/**
	 * Don't allow the context for remote connections.
	 * @throws ContextException if this context cannot be unexported
	 */
	void unexport() throws ContextException;

	/**
	 * Listen for events coming from the context with the given name.
	 * @param source a context name
	 * @return true if this context listens
	 */
	boolean listen(String source);

	/**
	 * Destroy this context.
	 * @throws ContextException if it cannot be destroyed
	 */
	void destroy() throws ContextException;

}
