/**
 *	This file is part of org.ow2.wildcat.
 *
 *	wildcat is free software: you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	wildcat is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 */
package org.ow2.wildcat.hierarchy;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

import org.antlr.runtime.ANTLRStringStream;
import org.antlr.runtime.CommonTokenStream;
import org.antlr.runtime.RecognitionException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.wildcat.util.Messages;
import org.ow2.wildcat.util.PathLexer;
import org.ow2.wildcat.util.PathParser;

/**
 * A path in the wildcat hierarchy. Path have the form of url.
 */
public class Path implements Serializable {

	/**
	 * Id for serializable class.
	 */
	private static final long serialVersionUID = 4106680673994638328L;

	/**
	 * Logger.
	 */
	private static transient Log logger = LogFactory.getLog(Path.class);

	private boolean absolute;
	private String hostname;
	private final List<String> resources;
	private String attribute;

	/**
	 * Constant Path for <b>/</b>.
	 */
	public static Path ROOT = new Path(true, null, new ArrayList<String>(),
			null);

	/**
	 * Constant Path for <b>null://</b>
	 */
	public static Path NULL = new Path(true, "null", new ArrayList<String>(),
			null);

	/**
	 * @param absolute
	 * @param hostname
	 * @param resources
	 * @param attribute
	 */
	private Path(final boolean absolute, final String hostname, final List<String> resources,
			final String attribute) {
		super();
		this.absolute = absolute;
		this.hostname = hostname;
		this.resources = resources;
		this.attribute = attribute;
	}

	/**
	 * Convenient constructors.
	 *
	 * Constructs a path from its string representation
	 *
	 * @param path
	 * @throws MalformedPathException
	 *             if the path is malformed
	 */
	public Path(final String path) throws MalformedPathException {
		logger.debug(Messages.message(Path.class.getName() + ".createString",
				path));

		PathLexer lexer = new PathLexer(new ANTLRStringStream(path));
		CommonTokenStream tokens = new CommonTokenStream(lexer);
		PathParser parser = new PathParser(tokens);
		PathParser.path_return pr = null;
		try {
			pr = parser.path();
		} catch (RecognitionException e) {
			logger.warn(Messages.message(
					Path.class.getName() + ".createString", path), e);
			throw new MalformedPathException("parsing failled");
		}

		this.absolute = pr.abs;
		this.hostname = pr.h;
		this.resources = pr.r;
		this.attribute = pr.a;
	}

	/**
	 * Deep copy constructor
	 *
	 * @param path
	 */
	public Path(final Path path) {

		this.hostname = path.hostname;
		this.absolute = path.absolute;
		this.resources = new ArrayList<String>();
		for (int i = 0; i < path.resources.size(); i++) {
			this.resources.add(path.resources.get(i));
		}
		this.attribute = path.attribute;
	}

	/**
	 * Creates a new path by appending 'name' as resource to current path
	 *
	 * @param name
	 * @return a new path with 'name' as resource added at the end of current
	 *         path
	 */
	public Path appendResource(final String name) {
		try {
			this.expectResource();

			Path n = new Path(this);
			n.resources.add(name);
			return n;
		} catch (MalformedPathException e) {
			return null;
		}
	}

	/**
	 * Creates a new path by appending 'name' as attribute name to current path
	 *
	 * assumes 'name' to be a valid attribute name
	 *
	 * @param name
	 * @return a copy of this path with 'name' as attribute name.
	 */
	public Path appendAttribute(final String name) {
		try {
			this.expectResource();
			Path n = new Path(this);
			n.attribute = name;
			return n;
		} catch (MalformedPathException e) {
			return null;
		}
	}

	/**
	 * Creates a new Path by appending suffix to 'this'.
	 *
	 * @param suffix
	 *            suffix to append
	 * @return a new Path
	 */
	public Path append(final Path suffix) {
		try {
			Path p = new Path(this);
			p.expectResource();

			if (suffix == null) {
				return p;
			} else {
				suffix.expectRelative();
				p.resources.addAll(suffix.resources);
				p.attribute = suffix.attribute;
				return p;
			}
		} catch (MalformedPathException e) {
			return null;
		}
	}

	/**
	 * Partially rewritte current path, changing oprefix in 'this' to nprefix
	 *
	 * @param oprefix
	 *            the prefix of current path to delete
	 * @param nprefix
	 *            the new prefix
	 * @return rewritted path
	 */
	public Path replacePrefix(final Path oprefix, final Path nprefix) {
		if (!this.isPrefix(oprefix)) {
			return null;
		}

		Path p = new Path(this);
		p.hostname = null;
		p.absolute = false;
		for (int i = 0; i < oprefix.resources.size(); i++) {
			p.resources.remove(0);
		}
		return nprefix.append(p);
	}

	/**
	 * Canonicalize current path
	 *
	 * @return a new Path canonicalized
	 */
	public Path canonicalize() {
		boolean abs = this.isAbsolute();
		String h = this.hostname;
		List<String> r = new ArrayList<String>();
		String a = this.attribute;

		for (String s : this.resources) {
			if (".".equals(s)) {

			} else if ("..".equals(s)) {
				if (r.size() > 0) {
					if ("..".equals(r.get(r.size() - 1))) {
						r.add("..");
					} else {
						r.remove(r.size() - 1);
					}
				} else {
					if (!abs) {
						r.add("..");
					}
				}
			} else {
				r.add(s);
			}
		}
		return new Path(abs, h, r, a);
	}

	/**
	 * Creates a new path by stripping the first resource as attribute name to
	 * current path. Only set absolute to false when the path is absolute.
	 *
	 * @return same as getRelativePart if path is absolute, else a copy of
	 *         current path without the first resource
	 */
	public Path stripTop() {
		try {
			Path p = new Path(this);
			if (p.isAbsolute()) {
				p.absolute = false;
				p.hostname = null;
			} else {
				p.resources.remove(0);
			}
			return p;
		} catch (Exception e) {
			return null;
		}
	}

	/**
	 * Creates a new path by stripping host and starting '/'
	 *
	 * @return a copy of current path wihtout host and starting '/'
	 */
	public Path getRelativePart() {
		if (!this.hasAttribute() && !this.hasResource()) {
			return null;
		}

		Path p = new Path(this);
		p.hostname = null;
		p.absolute = false;
		return p;
	}

	/**
	 * @return the hostname, <b>null</b> if none is explicitly given
	 */
	public String getHostname() {
		return this.hostname;
	}

	public void setHostName(final String hostname) {
		this.hostname = hostname;
	}

	/**
	 * @return the attribute name, <b>null</b> if none
	 */
	public String getAttributeName() {
		return this.attribute;
	}

	/**
	 * @return the first resource name of the path
	 */
	public String getFirstResource() {
		return this.resources.get(0);
	}

	/**
	 * @return <b>true</b> if path is absolute (it start with '/' or if
	 *         hasExplicitHost. is true), <b>false</b> otherwise.
	 */
	public boolean isAbsolute() {
		return this.absolute || this.hasExplicitHost();
	}

	/**
	 * @return <b>true</b> if path has an explicitly reference an host ('self'
	 *         or any other hostname) <b>false</b> otherwise.
	 */
	public boolean hasExplicitHost() {
		return hostname != null;
	}

	/**
	 * @return <b>true</b> if path points an attribute, else <b>false</b>
	 */
	public boolean hasAttribute() {
		return this.attribute != null;
	}

	/**
	 * @return <b>true<b> if path contains at least one resource
	 */
	public boolean hasResource() {
		return this.resources != null && this.resources.size() != 0;
	}

	/**
	 * @return <b>true</b> if path only contains an attribute name, else
	 *         <b>false</b>
	 */
	public boolean isOnlyAttribute() {
		return !this.isAbsolute()
		&& (this.resources == null || this.resources.size() == 0)
		&& this.hasAttribute();
	}

	/**
	 * @return <b>true</b> if path is only a resource name, else <b>false</b>
	 */
	public boolean isOnlyResource() {
		return !this.isAbsolute() && !this.hasAttribute()
		&& this.resources.size() == 1;
	}

	/**
	 * Tests if path is canonical.
	 *
	 * @return <b>true</b> if no resource is a relative reference ("." or "..")
	 */
	public boolean isCanonical() {
		if (this.resources != null) {
			for (String s : this.resources) {
				if (".".equals(s) || "..".equals(s)) {
					return false;
				}
			}
		}
		return true;
	}

	/**
	 * Tests if path contain pattern elements.
	 *
	 * @return <b>true</b> if a resource or the attribute equal "*", else
	 *         <b>false</b>
	 */
	public boolean isPattern() {
		if (this.resources != null) {
			for (String s : this.resources) {
				if ("*".equals(s)) {
					return true;
				}
			}
		}
		if ("*".equals(this.attribute)) {
			return true;
		}
		return false;
	}

	/**
	 * Test wether argument is a prefix of current path
	 *
	 * @param prefix
	 * @return <b>true</b>if prefix is a prefix of current path, else <b>false</b>
	 */
	public boolean isPrefix(final Path prefix) {
		try {
			prefix.expectAbsolute();
			prefix.expectResource();
			prefix.expectCanonical();
			prefix.expectFinite();

			if (this.resources.size() < prefix.resources.size()) {
				return false;
			}

			for (int i = 0; i < prefix.resources.size(); i++) {
				if (!prefix.resources.get(i).equals(this.resources.get(i))) {
					return false;
				}
			}

			return this.hostname.equals(prefix.hostname);
		} catch (Exception e) {
			return false;
		}
	}

	/**
	 * @param hostname
	 * @return <b>true</b> if is valid host name, else <b>false</b>.
	 */
	public static boolean isValidHostname(final String hostname) {
		return isValidID(hostname);
	}

	/**
	 * @param resourceName
	 * @return <b>true</b> if is valid resource name, else <b>false</b>.
	 */
	public static boolean isValidResourceName(final String resourceName) {
		return isValidID(resourceName);
	}

	/**
	 * @param attributeName
	 * @return <b>true</b> if is valid attribute name, else <b>false</b>.
	 */
	public static boolean isValidAttributeName(final String attributeName) {
		return isValidID(attributeName);
	}

	private static boolean isValidID(final String atom) {
		return Pattern.matches("[_a-zA-Z][a-zA-Z_0-9]*", atom);
	}

	/**
	 * Does nothing.
	 *
	 * @throws MalformedPathException
	 *             if path is not canonical
	 */
	public void expectCanonical() throws MalformedPathException {
		if (!this.isCanonical()) {
			throw new MalformedPathException("Canonical path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/**
	 * Does nothing.
	 *
	 * @throws MalformedPathException
	 *             if path is a pattern
	 */
	public void expectFinite() throws MalformedPathException {
		if (this.isPattern()) {
			throw new MalformedPathException("Finite path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/**
	 * Does nothing.
	 *
	 * @throws MalformedPathException
	 *             if path is not absolute
	 */
	public void expectAbsolute() throws MalformedPathException {
		if (!this.isAbsolute()) {
			throw new MalformedPathException("Absolute path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/**
	 * Does nothing.
	 *
	 * @throws MalformedPathException
	 *             if path is not relative
	 */
	public void expectRelative() throws MalformedPathException {
		if (this.isAbsolute()) {
			throw new MalformedPathException("Relative path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/**
	 * Does nothing.
	 *
	 * @throws MalformedPathException
	 *             if path denotes an attribute
	 */
	public void expectResource() throws MalformedPathException {
		if (this.hasAttribute()) {
			throw new MalformedPathException("Resource path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/**
	 * Does nothing
	 *
	 * @throws MalformedPathException
	 *             if path not denotes an attribute
	 */
	public void expectAttribute() throws MalformedPathException {
		if (!this.hasAttribute()) {
			throw new MalformedPathException("Attribute path expected got \'"
					+ this.toString() + "\'");
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		String c = "";
		boolean first = true;

		if (this.hostname != null) {
			c += this.hostname + "://";
		} else if (this.absolute) {
			c += "self://";
		}

		for (String r : this.resources) {
			if (first) {
				c += r;
				first = false;
			} else {
				c += "/" + r;
			}
		}

		if (this.hasAttribute()) {
			c += "#" + this.attribute;
		}

		return c;
	}
}
