/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 * --------------------------------------------------------------------------
 * $Id: QueryAttribute.java 366 2008-12-10 18:50:44Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.hierarchy.attribute;

import java.io.Serializable;

import net.esper.client.UpdateListener;
import net.esper.event.EventBean;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.ow2.wildcat.Context;
import org.ow2.wildcat.Query;
import org.ow2.wildcat.event.WAttributeEvent;
import org.ow2.wildcat.hierarchy.Mountable;
import org.ow2.wildcat.hierarchy.Path;
import org.ow2.wildcat.util.Messages;

/**
 * An attribute holding the result of a query.
 */
public class QueryAttribute implements Attribute, UpdateListener, Serializable {

    /**
     * Id for serializable class.
     */
    private static final long serialVersionUID = 5484145491004179031L;

    /**
     * Logger.
     */
    private static Log logger = LogFactory.getLog(QueryAttribute.class);

    private final String queryString;
    private Query query;
    private Object value;
    private Context context;
    private Path path;

    /**
     * Constructor.
     *
     * @param queryString
     *            the associated query
     */
    public QueryAttribute(final String queryString) {
        this.queryString = queryString;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.attribute.Attribute#getValue()
     */
    public Object getValue() {
        return this.value;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.attribute.Attribute#setValue(java.lang.Object)
     */
    public Object setValue(final Object value) {
        return this.value;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#isMount()
     */
    public boolean isMount() {
        return this.context != null && this.path != null;
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#mount(org.ow2.wildcat.Context,
     *      org.ow2.wildcat.hierarchy.Path)
     */
    public void mount(final Context context, final Path mountPoint) {
        logger.debug(Messages.message(Mountable.class.getName() + ".mount",
                mountPoint));

        if (this.isMount()) {
            logger.info(Messages.message(Mountable.class.getName()
                    + ".alreadymount", mountPoint));
        } else {
            this.context = context;
            this.path = mountPoint;

            this.query = this.context.registerListeners(this.queryString, this);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see org.ow2.wildcat.hierarchy.Mountable#unMount()
     */
    public void unMount() {
        logger.debug(Messages.message(Mountable.class.getName() + ".unmount",
                (Object[]) null));

        if (this.isMount() && this.query != null) {
            this.context.destroyQuery(this.query);
        }
        query = null;
        this.context = null;
        this.path = null;
    }

    /*
     * (non-Javadoc)
     *
     * @see net.esper.client.UpdateListener#update(net.esper.event.EventBean[],
     *      net.esper.event.EventBean[])
     */
    public void update(final EventBean[] arg0, final EventBean[] arg1) {
        if (arg0 == null || arg0.length == 0) {
            this.value = null;
        } else if (arg0.length == 1) {
            this.value = arg0[0].getUnderlying();
        } else {
            Object[] oa = new Object[arg0.length];
            for (int i = 0; i < arg0.length; i++) {
                oa[i] = arg0[i].getUnderlying();
            }
            this.value = oa;
        }
        if (this.isMount()) {
            this.context.emitEvent(new WAttributeEvent(this.path, this.value));
        }
    }

}
