/**
 * WildCAT: A Generic Framework for Context-Aware Applications.
 * Copyright (C) 2008 Bull S.A.S.
 * Copyright (C) 2008 EMN
 * Contact: wildcat@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Authors:
 *		- Pierre-Charles David (initial)
 *		- Nicolas Loriant
 * --------------------------------------------------------------------------
 * $Id: Resource.java 350 2008-11-10 09:41:14Z loris $
 * --------------------------------------------------------------------------
 */
package org.ow2.wildcat.hierarchy.resource;

import java.util.Set;

import org.ow2.wildcat.hierarchy.Mountable;
import org.ow2.wildcat.hierarchy.Path;
import org.ow2.wildcat.hierarchy.attribute.Attribute;

/**
 * Define a resource holding some attributes and other resources.
 */
public interface Resource extends Mountable {

    /**
     * Attach an attribute at specified location.
     *
     * @param attribute
     *            path of the Attribute
     * @param attr
     *            the Attribute
     * @throws ResourceException if the attribute cannot be attached
     */
    void attachAttribute(Path attribute, Attribute attr) throws ResourceException;

    /**
     * Create an attribute at specified location.
     *
     * @param attribute
     *            path of the attribute
     * @param value
     *            attribute initial value
     * @throws ResourceException if the attribute cannot be created
     */
    void createAttribute(Path attribute, Object value) throws ResourceException;

    /**
     * Return the attribute at specified location.
     * @param path path of the attribute
     * @return the attribute
     * @throws ResourceException if the attribute cannot be retrieved
     */
    Attribute getAttribute(Path path) throws ResourceException;

    /**
     * Detach an attribute at specified location.
     * @param attribute path of the attribute
     * @throws ResourceException if the attribute cannot be detached
     */
    void detachAttribute(Path attribute) throws ResourceException;

    /**
     * Fetch attribute value.
     *
     * @param path
     *            attribute's path
     * @return the value of the attribute
     * @throws ResourceException  if the attribute cannot be accessed
     */
    Object getValue(Path path) throws ResourceException;

    /**
     * Set Attribute's value.
     *
     * @param path
     *            path of the Attribute
     * @param value
     *            new value of the Attribute
     * @return previous value of the Attribute
     * @throws ResourceException if the attribute cannot be accessed
     */
    Object setValue(Path path, Object value) throws ResourceException;

    /**
     * Creates resources along a path
     *
     * @param resource
     *            path of resource
     * @throws ResourceException if the resource cannot be created
     */
    void createResource(Path resource) throws ResourceException;

    /**
     * Attach Resource at specified location.
     *
     * @param resource
     *            path of the Resource
     * @param res
     *            the Resource
     * @throws ResourceException if the resource cannot be attached
     */
    void attachResource(Path resource, Resource res) throws ResourceException;

    /**
     * Detach a resource at specified location.
     * @param resource path of the Resource
     * @throws ResourceException  if the resource cannot be detached
     */
    void detachResource(Path resource) throws ResourceException;

    /**
     * Return the resource at specified location.
     * @param res  path of the Resource
     * @return the resource at specified location.
     * @throws ResourceException if the resource cannot be accessed
     */
    Resource getResource(Path res) throws ResourceException;

    /**
     * List resources and attributes.
     *
     * @param resource
     *            indicates the resource to inspect, or 'this' if <b>null</b>
     * @return A set of String, attribute's names preceded with '#'
     * @throws ResourceException if the resource cannot be accessed
     */
    Set<String> list(Path resource) throws ResourceException;

}
